<?php
// Start session
session_start();

// Security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include QR code libraries
require_once 'QRCodeStyling.php';

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    generateQRCode();
}

// Handle download requests
if (isset($_GET['download'])) {
    downloadQRCode();
}

function generateQRCode() {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['qr_error'] = 'Security validation failed';
        return;
    }

    // Get input data
    $qrType = $_POST['qr_type'] ?? 'url';
    $options = [];

    // Build data based on QR type
    $data = buildQRData($qrType);

    if (!$data) {
        $_SESSION['qr_error'] = 'No data provided';
        return;
    }

    // Validate data length for QR code
    if (strlen($data) > 2953) {
        $_SESSION['qr_error'] = 'Data too long for QR code (max 2953 characters)';
        return;
    }

    // Basic options
    $options['data'] = $data;
    $options['width'] = 300;
    $options['height'] = 300;
    $options['margin'] = (int) ($_POST['margin'] ?? 10);

    // QR options
    $options['qrOptions'] = [
        'typeNumber' => 0,
        'mode' => null,
        'errorCorrectionLevel' => $_POST['ecc'] ?? 'Q'
    ];

    // Dots options
    $dotColorType = $_POST['dot_color_type'] ?? 'single';
    $options['dotsOptions'] = [
        'type' => $_POST['dot_type'] ?? 'square'
    ];

    if ($dotColorType === 'gradient') {
        $options['dotsOptions']['gradient'] = [
            'type' => $_POST['dot_grad_type'] ?? 'linear',
            'rotation' => 45,
            'colorStops' => [
                ['offset' => 0, 'color' => $_POST['dot_grad_1'] ?? '#4267b2'],
                ['offset' => 1, 'color' => $_POST['dot_grad_2'] ?? '#1a2a6c']
            ]
        ];
    } else {
        $options['dotsOptions']['color'] = $_POST['dot_color'] ?? '#000000';
    }

    // Background options
    $options['backgroundOptions'] = [
        'color' => (isset($_POST['bg_transparent']) && $_POST['bg_transparent'] === '1') ? 'transparent' : ($_POST['bg_color'] ?? '#FFFFFF')
    ];

    // Corners options
    $options['cornersSquareOptions'] = [
        'type' => $_POST['corner_sq_type'] ?? 'square',
        'color' => $_POST['corner_sq_color'] ?? '#000000'
    ];

    $options['cornersDotOptions'] = [
        'type' => $_POST['corner_dot_type'] ?? 'square',
        'color' => $_POST['corner_dot_color'] ?? '#000000'
    ];

    // Image options
    if (isset($_FILES['logo_file']) && $_FILES['logo_file']['error'] === UPLOAD_ERR_OK) {
        $allowedImageTypes = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/svg+xml' => 'svg'
        ];

        $validationErrors = validateUploadedFile($_FILES['logo_file'], $allowedImageTypes, 5 * 1024 * 1024);

        if (empty($validationErrors)) {
            $uploadDir = 'uploads/';
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Generate secure filename
            $fileExt = pathinfo($_FILES['logo_file']['name'], PATHINFO_EXTENSION);
            $fileName = uniqid('logo_', true) . '.' . $fileExt;
            $uploadPath = $uploadDir . $fileName;

            if (move_uploaded_file($_FILES['logo_file']['tmp_name'], $uploadPath)) {
                // Verify it's actually an image
                $imageInfo = getimagesize($uploadPath);
                if ($imageInfo !== false) {
                    $options['image'] = $uploadPath;
                    $options['imageOptions'] = [
                        'hideBackgroundDots' => (isset($_POST['hide_dots']) && $_POST['hide_dots'] === '1'),
                        'imageSize' => (float) ($_POST['logo_size'] ?? 0.4),
                        'margin' => (int) ($_POST['logo_margin'] ?? 10)
                    ];
                } else {
                    unlink($uploadPath); // Remove invalid file
                    $_SESSION['qr_error'] = 'Invalid image file uploaded';
                }
            }
        } else {
            $_SESSION['qr_error'] = implode(', ', $validationErrors);
        }
    }

    // Generate QR code and store in session
    try {
        $qr = new QRCodeStyling($options);
        $_SESSION['qr_options'] = $options;
        $_SESSION['qr_data'] = $data;

        // Generate preview image
        $previewData = $qr->getRawData('png');
        $_SESSION['qr_preview'] = 'data:image/png;base64,' . base64_encode($previewData);

    } catch (Exception $e) {
        $_SESSION['qr_error'] = $e->getMessage();
    }
}

function downloadQRCode() {
    if (!isset($_SESSION['qr_options'])) {
        die('No QR code to download');
    }

    $format = $_GET['format'] ?? 'png';
    $filename = 'qr_code_' . date('Ymd_His') . '.' . $format;

    try {
        $qr = new QRCodeStyling($_SESSION['qr_options']);

        if ($format === 'pdf') {
            // For PDF, generate PNG first
            $data = $qr->getRawData('png');

            // Create simple PDF
            $pdfContent = generatePDF($data);

            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . strlen($pdfContent));

            echo $pdfContent;
        } else {
            $data = $qr->getRawData($format);

            if ($format === 'svg') {
                header('Content-Type: image/svg+xml');
            } else {
                header('Content-Type: image/' . $format);
            }

            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . strlen($data));

            echo $data;
        }

        exit;

    } catch (Exception $e) {
        die('Error generating download: ' . htmlspecialchars($e->getMessage()));
    }
}

function generatePDF($qrData): string
{
    // Simple PDF header
    $pdf = "%PDF-1.4\n";
    $pdf .= "1 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Catalog\n";
    $pdf .= "/Pages 2 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n";

    // Pages object
    $pdf .= "2 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Pages\n";
    $pdf .= "/Kids [3 0 R]\n";
    $pdf .= "/Count 1\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n";

    // Page object
    $pdf .= "3 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Page\n";
    $pdf .= "/Parent 2 0 R\n";
    $pdf .= "/MediaBox [0 0 612 792]\n";
    $pdf .= "/Resources <<\n";
    $pdf .= "/XObject <<\n";
    $pdf .= "/Im1 4 0 R\n";
    $pdf .= ">>\n";
    $pdf .= ">>\n";
    $pdf .= "/Contents 5 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n";

    // Image object
    $pdf .= "4 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /XObject\n";
    $pdf .= "/Subtype /Image\n";
    $pdf .= "/Width 300\n";
    $pdf .= "/Height 300\n";
    $pdf .= "/ColorSpace /DeviceRGB\n";
    $pdf .= "/BitsPerComponent 8\n";
    $pdf .= "/Filter /FlateDecode\n";
    $pdf .= "/Length " . strlen($qrData) . "\n";
    $pdf .= ">>\n";
    $pdf .= "stream\n";
    $pdf .= $qrData . "\n";
    $pdf .= "endstream\n";
    $pdf .= "endobj\n";

    // Contents object
    $pdf .= "5 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Length 44\n";
    $pdf .= ">>\n";
    $pdf .= "stream\n";
    $pdf .= "q\n";
    $pdf .= "150 0 0 150 156 396 cm\n";
    $pdf .= "/Im1 Do\n";
    $pdf .= "Q\n";
    $pdf .= "endstream\n";
    $pdf .= "endobj\n";

    // Xref table
    $pdf .= "xref\n";
    $pdf .= "0 6\n";
    $pdf .= "0000000000 65535 f \n";
    $pdf .= "0000000010 00000 n \n";
    $pdf .= "0000000053 00000 n \n";
    $pdf .= "0000000145 00000 n \n";
    $pdf .= "0000000296 00000 n \n";
    $pdf .= "0000000450 00000 n \n";

    // Trailer
    $pdf .= "trailer\n";
    $pdf .= "<<\n";
    $pdf .= "/Size 6\n";
    $pdf .= "/Root 1 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "startxref\n";
    $pdf .= "578\n";
    $pdf .= "%%EOF";

    return $pdf;
}

function buildQRData($type) {
    switch ($type) {
        case 'url':
            return $_POST['input_url'] ?? '';

        case 'text':
            return $_POST['input_text'] ?? '';

        case 'vcard':
            return buildVCardData();

        case 'email':
            $to = $_POST['input_email_to'] ?? '';
            $subject = $_POST['input_email_sub'] ?? '';
            return "mailto:$to?subject=" . urlencode($subject);

        case 'sms':
            $phone = $_POST['input_sms_phone'] ?? '';
            $message = $_POST['input_sms_msg'] ?? '';
            return "sms:$phone?body=" . urlencode($message);

        case 'phone':
            $phone = $_POST['input_phone_only'] ?? '';
            return "tel:$phone";

        case 'app':
            return $_POST['input_app'] ?? '';

        case 'multi':
            return $_POST['input_multi'] ?? '';

        case 'pdf':
            if (isset($_FILES['input_pdf']) && $_FILES['input_pdf']['error'] === UPLOAD_ERR_OK) {
                $validationErrors = validateUploadedFile($_FILES['input_pdf'], ['application/pdf' => 'pdf'], 10 * 1024 * 1024);

                if (empty($validationErrors)) {
                    $uploadDir = 'uploads/pdfs/';
                    if (!file_exists($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }

                    $fileName = uniqid() . '_' . basename($_FILES['input_pdf']['name']);
                    $uploadPath = $uploadDir . $fileName;

                    if (move_uploaded_file($_FILES['input_pdf']['tmp_name'], $uploadPath)) {
                        return 'https://' . $_SERVER['HTTP_HOST'] . '/' . $uploadPath;
                    }
                }
            }
            return '';

        default:
            return $_POST['input_url'] ?? 'https://wala.guru';
    }
}

function buildVCardData() {
    $vCard = "BEGIN:VCARD\n";
    $vCard .= "VERSION:3.0\n";

    if (!empty($_POST['v_fn'])) {
        $vCard .= "FN:" . $_POST['v_fn'] . "\n";
        $vCard .= "N:" . $_POST['v_fn'] . ";" . ($_POST['v_ln'] ?? '') . ";;;\n";
    }

    if (!empty($_POST['v_org'])) {
        $vCard .= "ORG:" . $_POST['v_org'] . "\n";
    }

    if (!empty($_POST['v_phone'])) {
        $vCard .= "TEL:" . $_POST['v_phone'] . "\n";
    }

    if (!empty($_POST['v_email'])) {
        $vCard .= "EMAIL:" . $_POST['v_email'] . "\n";
    }

    $vCard .= "END:VCARD";
    return $vCard;
}

function validateUploadedFile($file, $allowedTypes = [], $maxSize = 5242880) {
    $errors = [];

    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'Upload failed with error code: ' . $file['error'];
        return $errors;
    }

    // Check file size
    if ($file['size'] > $maxSize) {
        $errors[] = 'File size exceeds maximum allowed size of ' . ($maxSize / 1024 / 1024) . 'MB';
    }

    // Check file type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    if (!empty($allowedTypes) && !in_array($mime, array_keys($allowedTypes))) {
        $errors[] = 'Invalid file type. Allowed types: ' . implode(', ', array_keys($allowedTypes));
    }

    // Additional security checks
    $filename = basename($file['name']);
    if (preg_match('/[^\w\.\-]/', $filename)) {
        $errors[] = 'Invalid filename';
    }

    return $errors;
}

// Get current preview image
$previewImage = $_SESSION['qr_preview'] ?? '';
$qrError = $_SESSION['qr_error'] ?? '';
$qrData = $_SESSION['qr_data'] ?? '';

// Clear error after displaying
if (isset($_SESSION['qr_error'])) {
    unset($_SESSION['qr_error']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QR Code Generator</title>

    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">

    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">

    <style>
        #qr-canvas {
            min-height: 300px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
        }

        #qr-canvas img {
            max-width: 100%;
            height: auto;
            border-radius: 4px;
        }

        .form-control-color {
            height: 38px;
            padding: 3px;
        }

        .qr-input-group {
            transition: all 0.3s ease;
        }

        .nav-tabs .nav-link {
            color: #6c757d;
            font-weight: 500;
        }

        .nav-tabs .nav-link.active {
            color: #0d6efd;
            border-bottom: 3px solid #0d6efd;
        }

        .form-range::-webkit-slider-thumb {
            background: #0d6efd;
        }

        .form-range::-moz-range-thumb {
            background: #0d6efd;
        }

        @media (max-width: 768px) {
            .sticky-top {
                position: static !important;
                margin-top: 20px;
            }
        }

        .color-value, .range-value {
            font-family: 'Courier New', monospace;
            font-size: 0.85em;
        }

        .file-info {
            min-height: 1.5em;
        }

        .tab-content {
            min-height: 400px;
        }

        @media (max-width: 992px) {
            .card-body.tab-content {
                padding: 1rem !important;
            }

            .nav-tabs .nav-link {
                padding: 0.5rem;
                font-size: 0.9rem;
            }
        }

        /* Loading animation for QR generation */
        .btn-loading {
            position: relative;
        }

        .btn-loading:after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            margin-top: -12px;
            margin-left: -12px;
            width: 24px;
            height: 24px;
            border: 2px solid rgba(255,255,255,0.3);
            border-top-color: white;
            border-radius: 50%;
            animation: btn-spinner 0.6s linear infinite;
        }

        @keyframes btn-spinner {
            to { transform: rotate(360deg); }
        }

        /* Smooth transitions */
        .qr-input-group, #dot-single-color, #dot-gradient-options {
            transition: opacity 0.3s ease, transform 0.3s ease;
        }

        /* Custom scrollbar for tab content */
        .tab-content::-webkit-scrollbar {
            width: 6px;
        }

        .tab-content::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 3px;
        }

        .tab-content::-webkit-scrollbar-thumb {
            background: #888;
            border-radius: 3px;
        }

        .tab-content::-webkit-scrollbar-thumb:hover {
            background: #555;
        }

        .border-top {
            border-top: 1px solid #dee2e6 !important;
        }
    </style>
</head>
<body>
<div class="container-fluid py-5 bg-light">
    <div class="container">

            <?php if (!empty($qrError)): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($qrError); ?></div>
            <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" action="">
            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">

            <div class="row">
                <div class="col-lg-8">
                    <div class="card shadow-sm border-0">
                        <div class="card-header bg-white pt-3">
                            <ul class="nav nav-tabs card-header-tabs" id="qrTabs" role="tablist">
                                <li class="nav-item"><a class="nav-link active" data-bs-toggle="tab" href="#content">Content</a></li>
                                <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#dots">Dots</a></li>
                                <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#corners">Corners</a></li>
                                <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#background">Background</a></li>
                                <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#image">Logo</a></li>
                            </ul>
                        </div>
                        <div class="card-body tab-content p-4">

                            <div class="tab-pane fade show active" id="content">
                                <label class="form-label fw-bold">Select QR Type</label>
                                <select id="qr-type-selector" name="qr_type" class="form-select mb-3" onchange="toggleInputFields()">
                                    <option value="url" <?php echo (($_POST['qr_type'] ?? 'url') === 'url') ? 'selected' : ''; ?>>Website URL</option>
                                    <option value="pdf" <?php echo (($_POST['qr_type'] ?? '') === 'pdf') ? 'selected' : ''; ?>>PDF File</option>
                                    <option value="multi" <?php echo (($_POST['qr_type'] ?? '') === 'multi') ? 'selected' : ''; ?>>Multi-URL (Smart Link)</option>
                                    <option value="vcard" <?php echo (($_POST['qr_type'] ?? '') === 'vcard') ? 'selected' : ''; ?>>Contact (vCard)</option>
                                    <option value="text" <?php echo (($_POST['qr_type'] ?? '') === 'text') ? 'selected' : ''; ?>>Plain Text</option>
                                    <option value="app" <?php echo (($_POST['qr_type'] ?? '') === 'app') ? 'selected' : ''; ?>>App Store / Play Store</option>
                                    <option value="sms" <?php echo (($_POST['qr_type'] ?? '') === 'sms') ? 'selected' : ''; ?>>SMS</option>
                                    <option value="email" <?php echo (($_POST['qr_type'] ?? '') === 'email') ? 'selected' : ''; ?>>Email</option>
                                    <option value="phone" <?php echo (($_POST['qr_type'] ?? '') === 'phone') ? 'selected' : ''; ?>>Phone Call</option>
                                </select>

                                <div id="input-fields-container">
                                    <!-- URL Group -->
                                    <div class="qr-input-group" id="group-url" style="display: <?php echo (($_POST['qr_type'] ?? 'url') === 'url') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Website URL</label>
                                        <input type="url" name="input_url" class="form-control" placeholder="https://example.com" value="<?php echo htmlspecialchars($_POST['input_url'] ?? 'https://wala.guru'); ?>">
                                    </div>

                                    <!-- PDF Group -->
                                    <div class="qr-input-group" id="group-pdf" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'pdf') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Upload PDF</label>
                                        <input type="file" name="input_pdf" class="form-control" accept=".pdf">
                                        <small class="text-muted">Dynamic QR: The code will link to this document on our server.</small>
                                    </div>

                                    <!-- Multi-URL Group -->
                                    <div class="qr-input-group" id="group-multi" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'multi') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Smart Redirect Link</label>
                                        <input type="url" name="input_multi" class="form-control" placeholder="Enter your smart link URL" value="<?php echo htmlspecialchars($_POST['input_multi'] ?? ''); ?>">
                                    </div>

                                    <!-- vCard Group -->
                                    <div class="qr-input-group" id="group-vcard" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'vcard') ? 'block' : 'none'; ?>;">
                                        <div class="row">
                                            <div class="col-6 mb-2"><label>First Name</label><input type="text" name="v_fn" class="form-control" value="<?php echo htmlspecialchars($_POST['v_fn'] ?? ''); ?>"></div>
                                            <div class="col-6 mb-2"><label>Last Name</label><input type="text" name="v_ln" class="form-control" value="<?php echo htmlspecialchars($_POST['v_ln'] ?? ''); ?>"></div>
                                            <div class="col-12 mb-2"><label>Organization</label><input type="text" name="v_org" class="form-control" value="<?php echo htmlspecialchars($_POST['v_org'] ?? ''); ?>"></div>
                                            <div class="col-12 mb-2"><label>Phone</label><input type="tel" name="v_phone" class="form-control" value="<?php echo htmlspecialchars($_POST['v_phone'] ?? ''); ?>"></div>
                                            <div class="col-12 mb-2"><label>Email</label><input type="email" name="v_email" class="form-control" value="<?php echo htmlspecialchars($_POST['v_email'] ?? ''); ?>"></div>
                                        </div>
                                    </div>

                                    <!-- Text Group -->
                                    <div class="qr-input-group" id="group-text" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'text') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Message</label>
                                        <textarea name="input_text" class="form-control" rows="3"><?php echo htmlspecialchars($_POST['input_text'] ?? ''); ?></textarea>
                                    </div>

                                    <!-- App Group -->
                                    <div class="qr-input-group" id="group-app" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'app') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">App Link (Store URL)</label>
                                        <input type="url" name="input_app" class="form-control" placeholder="https://apps.apple.com/..." value="<?php echo htmlspecialchars($_POST['input_app'] ?? ''); ?>">
                                    </div>

                                    <!-- SMS Group -->
                                    <div class="qr-input-group" id="group-sms" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'sms') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Phone Number</label>
                                        <input type="tel" name="input_sms_phone" class="form-control mb-2" value="<?php echo htmlspecialchars($_POST['input_sms_phone'] ?? ''); ?>">
                                        <label class="form-label">Message</label>
                                        <textarea name="input_sms_msg" class="form-control"><?php echo htmlspecialchars($_POST['input_sms_msg'] ?? ''); ?></textarea>
                                    </div>

                                    <!-- Email Group -->
                                    <div class="qr-input-group" id="group-email" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'email') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Recipient Email</label>
                                        <input type="email" name="input_email_to" class="form-control mb-2" value="<?php echo htmlspecialchars($_POST['input_email_to'] ?? ''); ?>">
                                        <label class="form-label">Subject</label>
                                        <input type="text" name="input_email_sub" class="form-control" value="<?php echo htmlspecialchars($_POST['input_email_sub'] ?? ''); ?>">
                                    </div>

                                    <!-- Phone Group -->
                                    <div class="qr-input-group" id="group-phone" style="display: <?php echo (($_POST['qr_type'] ?? '') === 'phone') ? 'block' : 'none'; ?>;">
                                        <label class="form-label">Phone Number</label>
                                        <input type="tel" name="input_phone_only" class="form-control" placeholder="+1234567890" value="<?php echo htmlspecialchars($_POST['input_phone_only'] ?? ''); ?>">
                                    </div>
                                </div>

                                <div class="row mt-4 pt-3 border-top">
                                    <div class="col-md-6">
                                        <label class="form-label">Margin</label>
                                        <input type="number" name="margin" class="form-control" value="<?php echo htmlspecialchars($_POST['margin'] ?? '10'); ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Error Correction</label>
                                        <select name="ecc" class="form-select">
                                            <option value="L" <?php echo (($_POST['ecc'] ?? 'Q') === 'L') ? 'selected' : ''; ?>>L (7%)</option>
                                            <option value="M" <?php echo (($_POST['ecc'] ?? 'Q') === 'M') ? 'selected' : ''; ?>>M (15%)</option>
                                            <option value="Q" <?php echo (($_POST['ecc'] ?? 'Q') === 'Q') ? 'selected' : ''; ?>>Q (25%)</option>
                                            <option value="H" <?php echo (($_POST['ecc'] ?? 'Q') === 'H') ? 'selected' : ''; ?>>H (30%)</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="dots">
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Dot Style</label>
                                        <select name="dot_type" class="form-select">
                                            <option value="square" <?php echo (($_POST['dot_type'] ?? 'square') === 'square') ? 'selected' : ''; ?>>Square</option>
                                            <option value="dots" <?php echo (($_POST['dot_type'] ?? '') === 'dots') ? 'selected' : ''; ?>>Dots</option>
                                            <option value="rounded" <?php echo (($_POST['dot_type'] ?? '') === 'rounded') ? 'selected' : ''; ?>>Rounded</option>
                                            <option value="extra-rounded" <?php echo (($_POST['dot_type'] ?? '') === 'extra-rounded') ? 'selected' : ''; ?>>Extra Rounded</option>
                                            <option value="classy" <?php echo (($_POST['dot_type'] ?? '') === 'classy') ? 'selected' : ''; ?>>Classy</option>
                                            <option value="classy-rounded" <?php echo (($_POST['dot_type'] ?? '') === 'classy-rounded') ? 'selected' : ''; ?>>Classy Rounded</option>
                                        </select>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Color Type</label>
                                        <select name="dot_color_type" class="form-select" onchange="toggleGradient('dot')">
                                            <option value="single" <?php echo (($_POST['dot_color_type'] ?? 'single') === 'single') ? 'selected' : ''; ?>>Single Color</option>
                                            <option value="gradient" <?php echo (($_POST['dot_color_type'] ?? '') === 'gradient') ? 'selected' : ''; ?>>Gradient</option>
                                        </select>
                                    </div>
                                </div>
                                <div id="dot-single-color" style="display: <?php echo (($_POST['dot_color_type'] ?? 'single') === 'single') ? 'block' : 'none'; ?>;">
                                    <label class="form-label">Color</label>
                                    <input type="color" name="dot_color" class="form-control form-control-color w-100" value="<?php echo htmlspecialchars($_POST['dot_color'] ?? '#4267b2'); ?>">
                                </div>
                                <div id="dot-gradient-options" style="display: <?php echo (($_POST['dot_color_type'] ?? '') === 'gradient') ? 'block' : 'none'; ?>;">
                                    <div class="row">
                                        <div class="col-6"><label>Color 1</label><input type="color" name="dot_grad_1" class="form-control form-control-color w-100" value="<?php echo htmlspecialchars($_POST['dot_grad_1'] ?? '#4267b2'); ?>"></div>
                                        <div class="col-6"><label>Color 2</label><input type="color" name="dot_grad_2" class="form-control form-control-color w-100" value="<?php echo htmlspecialchars($_POST['dot_grad_2'] ?? '#1a2a6c'); ?>"></div>
                                    </div>
                                    <label class="mt-2">Type</label>
                                    <select name="dot_grad_type" class="form-select mb-2">
                                        <option value="linear" <?php echo (($_POST['dot_grad_type'] ?? 'linear') === 'linear') ? 'selected' : ''; ?>>Linear</option>
                                        <option value="radial" <?php echo (($_POST['dot_grad_type'] ?? '') === 'radial') ? 'selected' : ''; ?>>Radial</option>
                                    </select>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="corners">
                                <h6 class="fw-bold">Corner Square</h6>
                                <select name="corner_sq_type" class="form-select mb-3">
                                    <option value="square" <?php echo (($_POST['corner_sq_type'] ?? 'square') === 'square') ? 'selected' : ''; ?>>Square</option>
                                    <option value="dot" <?php echo (($_POST['corner_sq_type'] ?? '') === 'dot') ? 'selected' : ''; ?>>Dot</option>
                                    <option value="extra-rounded" <?php echo (($_POST['corner_sq_type'] ?? '') === 'extra-rounded') ? 'selected' : ''; ?>>Extra Rounded</option>
                                </select>
                                <input type="color" name="corner_sq_color" class="form-control form-control-color mb-3 w-100" value="<?php echo htmlspecialchars($_POST['corner_sq_color'] ?? '#000000'); ?>">

                                <h6 class="fw-bold mt-4">Corner Dot</h6>
                                <select name="corner_dot_type" class="form-select mb-3">
                                    <option value="square" <?php echo (($_POST['corner_dot_type'] ?? 'square') === 'square') ? 'selected' : ''; ?>>Square</option>
                                    <option value="dot" <?php echo (($_POST['corner_dot_type'] ?? '') === 'dot') ? 'selected' : ''; ?>>Dot</option>
                                </select>
                                <input type="color" name="corner_dot_color" class="form-control form-control-color w-100" value="<?php echo htmlspecialchars($_POST['corner_dot_color'] ?? '#000000'); ?>">
                            </div>

                            <div class="tab-pane fade" id="background">
                                <label class="form-label">Background Color</label>
                                <input type="color" name="bg_color" class="form-control form-control-color w-100 mb-3" value="<?php echo htmlspecialchars($_POST['bg_color'] ?? '#ffffff'); ?>">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="bg_transparent" value="1" id="bg-transparent" <?php echo (isset($_POST['bg_transparent']) && $_POST['bg_transparent'] === '1') ? 'checked' : ''; ?>>
                                    <label class="form-check-label" for="bg-transparent">Transparent Background (SVG only)</label>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="image">
                                <label class="form-label">Upload Logo</label>
                                <input type="file" name="logo_file" class="form-control mb-3" accept="image/*">
                                <div class="row">
                                    <div class="col-md-6">
                                        <label class="form-label">Logo Size (%)</label>
                                        <input type="range" name="logo_size" class="form-range" min="0.1" max="0.5" step="0.05" value="<?php echo htmlspecialchars($_POST['logo_size'] ?? '0.4'); ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Logo Margin</label>
                                        <input type="number" name="logo_margin" class="form-control" value="<?php echo htmlspecialchars($_POST['logo_margin'] ?? '10'); ?>">
                                    </div>
                                </div>
                                <div class="form-check mt-3">
                                    <input class="form-check-input" type="checkbox" name="hide_dots" value="1" id="hide-dots" <?php echo (isset($_POST['hide_dots']) && $_POST['hide_dots'] === '1') ? 'checked' : ''; ?>>
                                    <label class="form-check-label" for="hide-dots">Hide dots behind logo</label>
                                </div>
                            </div>
                        </div>
                        <div class="card-footer bg-white">
                            <button type="submit" class="btn btn-primary btn-lg w-100">
                                <i class="fas fa-qrcode me-2"></i> Generate QR Code
                            </button>
                        </div>
                    </div>
                </div>

                <div class="col-lg-4">
                    <div class="card shadow border-0 sticky-top" style="top: 20px;">
                        <div class="card-body text-center py-5">
                            <div id="qr-canvas" class="mb-4">
                                <?php if ($previewImage): ?>
                                    <img src="<?php echo $previewImage; ?>" alt="QR Code" class="img-fluid" style="max-width: 100%;">
                                <?php elseif ($qrError): ?>
                                    <div class="alert alert-warning">Please fix errors and generate QR code</div>
                                <?php else: ?>
                                    <div class="text-muted">QR Code will appear here after generation</div>
                                <?php endif; ?>
                            </div>

                            <?php if ($previewImage): ?>
                                <div class="mb-3">
                                    <small class="text-muted">Data: <?php echo htmlspecialchars(substr($qrData, 0, 50)) . (strlen($qrData) > 50 ? '...' : ''); ?></small>
                                </div>

                                <div class="d-grid gap-2">
                                    <a href="?download=png" class="btn btn-dark btn-lg">Download PNG</a>
                                    <div class="btn-group">
                                        <a href="?download=svg" class="btn btn-outline-dark">SVG</a>
                                        <a href="?download=pdf" class="btn btn-outline-dark">PDF</a>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<!-- Font Awesome for icons -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/js/all.min.js"></script>

<script>
    function toggleInputFields() {
        const type = document.getElementById('qr-type-selector').value;

        // Hide all groups
        document.querySelectorAll('.qr-input-group').forEach(group => {
            group.style.display = 'none';
        });

        // Show selected group
        const selectedGroup = document.getElementById(`group-${type}`);
        if (selectedGroup) {
            selectedGroup.style.display = 'block';
        }

        updateQRDataLength();
    }

    function toggleGradient(type) {
        const colorType = document.querySelector(`select[name="${type}_color_type"]`).value;
        const singleDiv = document.getElementById(`${type}-single-color`);
        const gradientDiv = document.getElementById(`${type}-gradient-options`);

        if (colorType === 'gradient') {
            singleDiv.style.display = 'none';
            gradientDiv.style.display = 'block';
        } else {
            singleDiv.style.display = 'block';
            gradientDiv.style.display = 'none';
        }
    }

    // Initialize Bootstrap tabs
    document.addEventListener('DOMContentLoaded', function() {
        const tabEls = document.querySelectorAll('button[data-bs-toggle="tab"]');
        tabEls.forEach(tabEl => {
            tabEl.addEventListener('shown.bs.tab', function(event) {
                // Optional: Save active tab to sessionStorage
                sessionStorage.setItem('activeTab', event.target.getAttribute('href'));
            });
        });

        // Restore active tab if exists
        const activeTab = sessionStorage.getItem('activeTab');
        if (activeTab) {
            const tabTrigger = new bootstrap.Tab(document.querySelector(`[href="${activeTab}"]`));
            tabTrigger.show();
        }

        // Preview QR code generation (optional real-time preview)
        const generateBtn = document.querySelector('button[type="submit"]');
        if (generateBtn) {
            generateBtn.addEventListener('click', function(e) {
                // Optional: Add loading state
                generateBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i> Generating...';
                generateBtn.disabled = true;

                // Re-enable button after 5 seconds if form doesn't submit
                setTimeout(() => {
                    generateBtn.innerHTML = '<i class="fas fa-qrcode me-2"></i> Generate QR Code';
                    generateBtn.disabled = false;
                }, 5000);
            });
        }

        // Color picker value display enhancement
        const colorPickers = document.querySelectorAll('input[type="color"]');
        colorPickers.forEach(picker => {
            picker.addEventListener('input', function() {
                const parent = this.closest('.mb-3') || this.closest('.row');
                if (parent) {
                    let valueDisplay = parent.querySelector('.color-value');
                    if (!valueDisplay) {
                        valueDisplay = document.createElement('small');
                        valueDisplay.className = 'color-value text-muted ms-2';
                        this.parentNode.appendChild(valueDisplay);
                    }
                    valueDisplay.textContent = this.value;
                }
            });

            // Trigger initial display
            picker.dispatchEvent(new Event('input'));
        });

        // Range slider value display
        const rangeSliders = document.querySelectorAll('input[type="range"]');
        rangeSliders.forEach(slider => {
            slider.addEventListener('input', function() {
                const value = parseFloat(this.value) * 100;
                const parent = this.closest('.col-md-6') || this.closest('div');
                if (parent) {
                    let valueDisplay = parent.querySelector('.range-value');
                    if (!valueDisplay) {
                        valueDisplay = document.createElement('small');
                        valueDisplay.className = 'range-value text-muted ms-2';
                        this.parentNode.appendChild(valueDisplay);
                    }
                    valueDisplay.textContent = Math.round(value) + '%';
                }
            });

            // Trigger initial display
            slider.dispatchEvent(new Event('input'));
        });

        // Form validation
        const form = document.querySelector('form');
        if (form) {
            form.addEventListener('submit', function(e) {
                const qrType = document.getElementById('qr-type-selector').value;
                let isValid = true;
                let errorMessage = '';

                // Validate based on QR type
                switch (qrType) {
                    case 'url':
                        const urlInput = document.querySelector('input[name="input_url"]');
                        if (urlInput && !urlInput.value.trim()) {
                            isValid = false;
                            errorMessage = 'Please enter a website URL';
                        }
                        break;
                    case 'text':
                        const textInput = document.querySelector('textarea[name="input_text"]');
                        if (textInput && !textInput.value.trim()) {
                            isValid = false;
                            errorMessage = 'Please enter text content';
                        }
                        break;
                    case 'vcard':
                        const fnInput = document.querySelector('input[name="v_fn"]');
                        if (fnInput && !fnInput.value.trim()) {
                            isValid = false;
                            errorMessage = 'Please enter at least a first name for the contact';
                        }
                        break;
                    case 'email':
                        const emailInput = document.querySelector('input[name="input_email_to"]');
                        if (emailInput && !emailInput.value.trim()) {
                            isValid = false;
                            errorMessage = 'Please enter recipient email';
                        }
                        break;
                    case 'sms':
                        const phoneInput = document.querySelector('input[name="input_sms_phone"]');
                        if (phoneInput && !phoneInput.value.trim()) {
                            isValid = false;
                            errorMessage = 'Please enter phone number';
                        }
                        break;
                    case 'phone':
                        const phoneOnlyInput = document.querySelector('input[name="input_phone_only"]');
                        if (phoneOnlyInput && !phoneOnlyInput.value.trim()) {
                            isValid = false;
                            errorMessage = 'Please enter phone number';
                        }
                        break;
                    case 'pdf':
                        const pdfInput = document.querySelector('input[name="input_pdf"]');
                        if (pdfInput && !pdfInput.files.length) {
                            isValid = false;
                            errorMessage = 'Please select a PDF file';
                        }
                        break;
                }

                if (!isValid) {
                    e.preventDefault();
                    alert(errorMessage);
                    return false;
                }
            });
        }

        // File input preview for PDF and logo
        const fileInputs = document.querySelectorAll('input[type="file"]');
        fileInputs.forEach(input => {
            input.addEventListener('change', function() {
                if (this.files && this.files[0]) {
                    const fileName = this.files[0].name;
                    const fileSize = (this.files[0].size / 1024 / 1024).toFixed(2); // MB

                    // Create file info display
                    let fileInfo = this.parentNode.querySelector('.file-info');
                    if (!fileInfo) {
                        fileInfo = document.createElement('div');
                        fileInfo.className = 'file-info small mt-1';
                        this.parentNode.appendChild(fileInfo);
                    }

                    if (this.accept.includes('.pdf')) {
                        if (this.files[0].size > 10 * 1024 * 1024) { // 10MB limit
                            fileInfo.innerHTML = `<span class="text-danger">File too large (max 10MB)</span>`;
                            this.value = '';
                        } else {
                            fileInfo.innerHTML = `<span class="text-success">${fileName} (${fileSize} MB)</span>`;
                        }
                    } else if (this.accept.includes('image')) {
                        if (this.files[0].size > 5 * 1024 * 1024) { // 5MB limit
                            fileInfo.innerHTML = `<span class="text-danger">Image too large (max 5MB)</span>`;
                            this.value = '';
                        } else {
                            // Check if it's an image
                            const img = new Image();
                            img.onload = function() {
                                fileInfo.innerHTML = `<span class="text-success">${fileName} (${fileSize} MB, ${this.width}×${this.height})</span>`;
                            };
                            img.onerror = function() {
                                fileInfo.innerHTML = `<span class="text-danger">Invalid image file</span>`;
                                input.value = '';
                            };
                            img.src = URL.createObjectURL(this.files[0]);
                        }
                    }
                }
            });
        });

        // Add QR data length indicator
        const textInputs = document.querySelectorAll('input[type="text"], input[type="url"], textarea');
        textInputs.forEach(input => {
            if (input.name.includes('input_') || input.name.includes('v_')) {
                input.addEventListener('input', function() {
                    updateQRDataLength();
                });
            }
        });

        function updateQRDataLength() {
            const qrType = document.getElementById('qr-type-selector').value;
            let data = '';

            switch (qrType) {
                case 'url':
                    data = document.querySelector('input[name="input_url"]')?.value || '';
                    break;
                case 'text':
                    data = document.querySelector('textarea[name="input_text"]')?.value || '';
                    break;
                case 'multi':
                    data = document.querySelector('input[name="input_multi"]')?.value || '';
                    break;
                case 'app':
                    data = document.querySelector('input[name="input_app"]')?.value || '';
                    break;
                case 'vcard':
                    const fn = document.querySelector('input[name="v_fn"]')?.value || '';
                    const ln = document.querySelector('input[name="v_ln"]')?.value || '';
                    const org = document.querySelector('input[name="v_org"]')?.value || '';
                    const phone = document.querySelector('input[name="v_phone"]')?.value || '';
                    const email = document.querySelector('input[name="v_email"]')?.value || '';
                    data = `FN:${fn} N:${fn};${ln} ORG:${org} TEL:${phone} EMAIL:${email}`;
                    break;
                case 'email':
                    const emailTo = document.querySelector('input[name="input_email_to"]')?.value || '';
                    const subject = document.querySelector('input[name="input_email_sub"]')?.value || '';
                    data = `mailto:${emailTo}?subject=${subject}`;
                    break;
                case 'sms':
                    const smsPhone = document.querySelector('input[name="input_sms_phone"]')?.value || '';
                    const smsMsg = document.querySelector('textarea[name="input_sms_msg"]')?.value || '';
                    data = `sms:${smsPhone}?body=${smsMsg}`;
                    break;
                case 'phone':
                    data = document.querySelector('input[name="input_phone_only"]')?.value || '';
                    break;
            }

            let lengthIndicator = document.querySelector('#qr-data-length');
            if (!lengthIndicator) {
                const container = document.querySelector('#content .border-top');
                if (container) {
                    lengthIndicator = document.createElement('div');
                    lengthIndicator.id = 'qr-data-length';
                    lengthIndicator.className = 'small text-muted mt-2';
                    container.parentNode.insertBefore(lengthIndicator, container);
                }
            }

            if (lengthIndicator) {
                const length = data.length;
                const maxLength = 2953; // Max for QR v40 with L ECC
                lengthIndicator.textContent = `Data length: ${length} characters (max: ${maxLength})`;

                if (length > maxLength) {
                    lengthIndicator.className = 'small text-danger mt-2';
                } else if (length > maxLength * 0.8) {
                    lengthIndicator.className = 'small text-warning mt-2';
                } else {
                    lengthIndicator.className = 'small text-muted mt-2';
                }
            }
        }

        // Initialize data length indicator
        updateQRDataLength();

        // Responsive adjustments
        function handleResponsive() {
            const qrCanvas = document.getElementById('qr-canvas');
            if (qrCanvas) {
                const width = qrCanvas.offsetWidth;
                if (width < 350) {
                    qrCanvas.style.padding = '10px';
                } else {
                    qrCanvas.style.padding = '20px';
                }
            }
        }

        window.addEventListener('resize', handleResponsive);
        handleResponsive();

        // Add keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + Enter to submit form
            if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
                const form = document.querySelector('form');
                if (form) {
                    form.submit();
                }
            }

            // Escape to clear form
            if (e.key === 'Escape') {
                const confirmClear = confirm('Clear all form data?');
                if (confirmClear) {
                    document.querySelector('form').reset();
                    updateQRDataLength();
                }
            }
        });
    });
</script>
</body>
</html>