// Main JavaScript for QRStuff Clone
$(document).ready(function() {
    // Initialize all components
    initializeApp();
});

// Main Initialization Function
function initializeApp() {
    // Set default content if empty
    initializeDefaults();

    // Initialize UI components
    initializeUI();

    // Initialize core functionality
    initializeCoreFeatures();

    // Initialize analytics if available
    if ($('#analyticsChart').length || $('#userStatsChart').length) {
        initializeCharts();
    }
}

// Set Default Values
function initializeDefaults() {
    try {
        // Set default URL if no content exists
        const urlInput = $('#url-input');
        const textInput = $('#text-input');

        if (!urlInput.val() && !textInput.val()) {
            urlInput.val('https://example.com');
            console.log('Set default URL content');
        }

        // Set active tab if none selected
        if (!$('.qr-type-tab.active').length) {
            $('.qr-type-tab[data-type="url"]').addClass('active');
            console.log('Set URL as active tab');
        }
    } catch (error) {
        console.error('Error initializing defaults:', error);
    }
}

// Initialize UI Components
function initializeUI() {
    try {
        // Initialize tooltips
        if ($('[data-bs-toggle="tooltip"]').length) {
            $('[data-bs-toggle="tooltip"]').tooltip();
        }

        // Initialize popovers
        if ($('[data-bs-toggle="popover"]').length) {
            $('[data-bs-toggle="popover"]').popover();
        }

        // Initialize color pickers
        if ($('#fg-color-preview').length && $('#bg-color-preview').length) {
            initializeColorPickers();
        }
    } catch (error) {
        console.error('Error initializing UI:', error);
    }
}

// Initialize Core Features
function initializeCoreFeatures() {
    try {
        // Initialize QR code preview
        if ($('#qr-code-display').length) {
            initializeQRPreview();
        }

        // Initialize file uploads
        if ($('#upload-logo-btn').length) {
            initializeFileUploads();
        }

        // Initialize form validation
        if ($('#register-form').length || $('#login-form').length) {
            initializeFormValidation();
        }
    } catch (error) {
        console.error('Error initializing core features:', error);
    }
}

// Color Picker Initialization
function initializeColorPickers() {
    if (typeof Pickr === 'undefined') {
        console.error('Pickr library not loaded');
        return;
    }

    try {
        // Foreground color picker
        const fgPicker = Pickr.create({
            el: '#fg-color-preview',
            theme: 'classic',
            default: '#000000',
            swatches: [
                '#000000', '#ffffff', '#4a6fa5', '#4fc3a1', '#ff6b6b',
                '#ffa726', '#66bb6a', '#42a5f5', '#7e57c2', '#ef5350'
            ],
            components: {
                preview: true,
                opacity: true,
                hue: true,
                interaction: {
                    hex: true,
                    rgba: true,
                    input: true,
                    save: true
                }
            }
        }).on('save', (color, instance) => {
            const hexColor = color.toHEXA().toString();
            $('#fg-color').val(hexColor);
            $('#fg-color-preview').css('backgroundColor', hexColor);
            updateQRPreview();
        }).on('clear', () => {
            $('#fg-color').val('#000000');
            $('#fg-color-preview').css('backgroundColor', '#000000');
            updateQRPreview();
        });

        window.fgColorPicker = fgPicker; // Store for later access

        // Background color picker
        const bgPicker = Pickr.create({
            el: '#bg-color-preview',
            theme: 'classic',
            default: '#ffffff',
            swatches: [
                '#ffffff', '#000000', '#f8f9fa', '#e9ecef', '#dee2e6',
                '#ced4da', '#adb5bd', '#6c757d', '#495057', '#212529'
            ],
            components: {
                preview: true,
                opacity: true,
                hue: true,
                interaction: {
                    hex: true,
                    rgba: true,
                    input: true,
                    save: true
                }
            }
        }).on('save', (color, instance) => {
            const hexColor = color.toHEXA().toString();
            $('#bg-color').val(hexColor);
            $('#bg-color-preview').css('backgroundColor', hexColor);
            updateQRPreview();
        }).on('clear', () => {
            $('#bg-color').val('#ffffff');
            $('#bg-color-preview').css('backgroundColor', '#ffffff');
            updateQRPreview();
        });

        window.bgColorPicker = bgPicker; // Store for later access

    } catch (error) {
        console.error('Error initializing color pickers:', error);
    }
}

// QR Preview Initialization with Debouncing
function initializeQRPreview() {
    if (typeof QRCode === 'undefined') {
        console.error('QRCode library not loaded');
        return;
    }

    // Debounced update function for better performance
    const debouncedUpdate = debounce(updateQRPreview, 300);

    // Update preview on input changes
    $('input, textarea, select').on('input change', function() {
        if ($(this).data('update-qr') !== false) {
            debouncedUpdate();
        }
    });

    // Initial preview update
    updateQRPreview();
}

// Update QR Preview
function updateQRPreview() {
    const qrDisplay = $('#qr-code-display');
    if (!qrDisplay.length) return;

    const content = getQRContent();
    if (!content || content.trim() === '') {
        qrDisplay.html('<div class="text-muted text-center p-4">Enter content to see preview</div>');
        return;
    }

    const size = parseInt($('#qr-size').val()) || 300;
    const margin = parseInt($('#qr-margin').val()) || 4;
    const fgColor = $('#fg-color').val() || '#000000';
    const bgColor = $('#bg-color').val() || '#ffffff';
    const scale = Math.min(1, 300 / size);
    const displaySize = size * scale;

    // Clear previous QR
    qrDisplay.empty();

    try {
        // Generate new QR code
        QRCode.toCanvas($('<canvas></canvas>')[0], content, {
            width: displaySize,
            margin: margin / 300 * displaySize,
            color: {
                dark: fgColor,
                light: bgColor
            },
            errorCorrectionLevel: $('#error-correction').val() || 'L'
        }, function(error, canvas) {
            if (error) {
                console.error('QR Code generation error:', error);
                qrDisplay.html('<div class="text-danger text-center p-4">Error generating preview</div>');
                return;
            }

            qrDisplay.append(canvas);

            // Add logo preview if exists
            const logoPath = $('#logo-path').val();
            const logoPreview = $('#logo-preview-img');
            if (logoPath && logoPreview.length && logoPreview.attr('src')) {
                addLogoToPreview(canvas, logoPreview.attr('src'));
            }
        });
    } catch (error) {
        console.error('Error generating QR preview:', error);
        qrDisplay.html('<div class="text-danger text-center p-4">Error generating preview</div>');
    }
}

// Add Logo to QR Preview
function addLogoToPreview(canvas, logoUrl) {
    const ctx = canvas.getContext('2d');
    const logoSize = canvas.width * 0.2; // 20% of QR size
    const x = (canvas.width - logoSize) / 2;
    const y = (canvas.height - logoSize) / 2;

    const logo = new Image();
    logo.onload = function() {
        // Draw white background for logo
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(x - 2, y - 2, logoSize + 4, logoSize + 4);

        // Draw logo
        ctx.drawImage(logo, x, y, logoSize, logoSize);
    };
    logo.src = logoUrl;
}

// Get QR Content with Validation
function getQRContent() {
    const activeTab = $('.qr-type-tab.active');
    if (activeTab.length === 0) return '';

    const type = activeTab.data('type');
    let content = '';

    switch(type) {
        case 'url':
            content = $('#url-input').val() || '';
            if (content && !content.match(/^https?:\/\//)) {
                content = 'http://' + content;
            }
            break;
        case 'text':
            content = $('#text-input').val() || '';
            break;
        case 'email':
            const to = $('#email-to').val() || '';
            const subject = $('#email-subject').val() || '';
            const body = $('#email-body').val() || '';
            if (to) {
                content = `mailto:${to}`;
                const params = [];
                if (subject) params.push(`subject=${encodeURIComponent(subject)}`);
                if (body) params.push(`body=${encodeURIComponent(body)}`);
                if (params.length > 0) content += '?' + params.join('&');
            }
            break;
        case 'sms':
            const phone = $('#sms-phone').val() || '';
            const message = $('#sms-message').val() || '';
            if (phone) {
                content = `smsto:${phone}`;
                if (message) content += `:${encodeURIComponent(message)}`;
            }
            break;
        case 'wifi':
            const ssid = $('#wifi-ssid').val() || '';
            const password = $('#wifi-password').val() || '';
            const encryption = $('#wifi-encryption').val() || 'WPA';
            const hidden = $('#wifi-hidden').is(':checked');

            if (ssid) {
                content = `WIFI:T:${encryption};S:${ssid};`;
                if (password) content += `P:${password};`;
                if (hidden) content += 'H:true;';
                content += ';';
            }
            break;
        case 'vcard':
            const name = $('#vcard-name').val() || '';
            const company = $('#vcard-company').val() || '';
            const phoneV = $('#vcard-phone').val() || '';
            const email = $('#vcard-email').val() || '';
            const website = $('#vcard-website').val() || '';
            const address = $('#vcard-address').val() || '';
            const note = $('#vcard-note').val() || '';

            if (name) {
                let vcard = "BEGIN:VCARD\nVERSION:3.0\n";
                vcard += `FN:${name}\n`;
                if (company) vcard += `ORG:${company}\n`;
                if (phoneV) vcard += `TEL:${phoneV}\n`;
                if (email) vcard += `EMAIL:${email}\n`;
                if (website) vcard += `URL:${website}\n`;
                if (address) vcard += `ADR:${address}\n`;
                if (note) vcard += `NOTE:${note}\n`;
                vcard += "END:VCARD";
                content = vcard;
            }
            break;
        default:
            content = '';
    }

    return content;
}

// File Upload Initialization
function initializeFileUploads() {
    // Logo upload
    $('#upload-logo-btn').on('click', function() {
        $('#logo-upload').click();
    });

    $('#logo-upload').on('change', function(e) {
        const file = e.target.files[0];
        if (!file) return;

        // Validate file
        if (!validateImageFile(file)) {
            showAlert('error', 'Please upload a valid image file (JPG, PNG, GIF, SVG) under 5MB.');
            $(this).val(''); // Clear file input
            return;
        }

        // Show preview
        showLogoPreview(file);

        // Upload to server if logged in
        if (isLoggedIn()) {
            uploadLogo(file);
        } else {
            // Store file locally for preview
            const reader = new FileReader();
            reader.onload = function(event) {
                $('#logo-path').val('local://' + file.name);
                updateQRPreview();
            };
            reader.readAsDataURL(file);
        }
    });

    // Remove logo
    $('#remove-logo-btn').on('click', function() {
        $('#logo-preview').hide();
        $('#logo-path').val('');
        $('#logo-upload').val('');
        updateQRPreview();
        showAlert('info', 'Logo removed');
    });
}

// Show Logo Preview
function showLogoPreview(file) {
    const reader = new FileReader();
    reader.onload = function(event) {
        $('#logo-preview-img').attr('src', event.target.result);
        $('#logo-preview').show();
        updateQRPreview();
    };
    reader.onerror = function() {
        showAlert('error', 'Failed to read image file');
        $('#logo-upload').val('');
    };
    reader.readAsDataURL(file);
}

// Validate Image File
function validateImageFile(file) {
    const maxSize = 5 * 1024 * 1024; // 5MB
    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/svg+xml'];

    if (file.size > maxSize) {
        showAlert('error', 'File size must be less than 5MB');
        return false;
    }

    if (!allowedTypes.includes(file.type.toLowerCase())) {
        showAlert('error', 'File type not supported. Please use JPG, PNG, GIF, or SVG.');
        return false;
    }

    return true;
}

// Upload Logo to Server
function uploadLogo(file) {
    const formData = new FormData();
    formData.append('logo', file);

    $.ajax({
        url: SITE_URL + 'api/v1/upload/logo',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        headers: getAuthHeaders(),
        beforeSend: function() {
            $('#upload-logo-btn').prop('disabled', true)
                .html('<i class="fas fa-spinner fa-spin me-1"></i> Uploading...');
        },
        success: function(response) {
            if (response.success) {
                $('#logo-path').val(response.data.path);
                showAlert('success', 'Logo uploaded successfully');
            } else {
                showAlert('error', response.error || 'Upload failed');
                $('#logo-upload').val('');
            }
        },
        error: function(xhr, status, error) {
            console.error('Logo upload error:', error);
            showAlert('error', 'Failed to upload logo. Please try again.');
            $('#logo-upload').val('');
        },
        complete: function() {
            $('#upload-logo-btn').prop('disabled', false)
                .html('<i class="fas fa-upload me-1"></i> Upload Logo');
        }
    });
}

// Form Validation Initialization
function initializeFormValidation() {
    // Registration form
    $('#register-form').on('submit', function(e) {
        e.preventDefault();

        const email = $('#register-email').val().trim();
        const password = $('#register-password').val();
        const confirmPassword = $('#register-confirm-password').val();

        // Validation
        if (!isValidEmail(email)) {
            showAlert('error', 'Please enter a valid email address');
            return;
        }

        if (password.length < 6) {
            showAlert('error', 'Password must be at least 6 characters');
            return;
        }

        if (!/[A-Z]/.test(password)) {
            showAlert('error', 'Password must contain at least one uppercase letter');
            return;
        }

        if (!/[0-9]/.test(password)) {
            showAlert('error', 'Password must contain at least one number');
            return;
        }

        if (password !== confirmPassword) {
            showAlert('error', 'Passwords do not match');
            return;
        }

        // Submit form
        submitRegistration(email, password);
    });

    // Login form
    $('#login-form').on('submit', function(e) {
        e.preventDefault();

        const email = $('#login-email').val().trim();
        const password = $('#login-password').val();

        if (!email || !password) {
            showAlert('error', 'Please enter email and password');
            return;
        }

        if (!isValidEmail(email)) {
            showAlert('error', 'Please enter a valid email address');
            return;
        }

        submitLogin(email, password);
    });

    // Forgot password form
    $('#forgot-password-form').on('submit', function(e) {
        e.preventDefault();

        const email = $('#forgot-email').val().trim();

        if (!isValidEmail(email)) {
            showAlert('error', 'Please enter a valid email address');
            return;
        }

        submitForgotPassword(email);
    });
}

// Email Validation
function isValidEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}

// Submit Registration
function submitRegistration(email, password) {
    const btn = $('#register-form button[type="submit"]');
    const originalText = btn.html();

    btn.prop('disabled', true)
        .html('<i class="fas fa-spinner fa-spin me-2"></i> Registering...');

    $.ajax({
        url: SITE_URL + 'api/v1/user/register',
        type: 'POST',
        data: JSON.stringify({
            email: email,
            password: password
        }),
        contentType: 'application/json',
        success: function(response) {
            if (response.success) {
                showAlert('success', 'Registration successful! Redirecting...');
                setTimeout(() => {
                    window.location.href = SITE_URL + 'dashboard.php';
                }, 1500);
            } else {
                showAlert('error', response.error || 'Registration failed');
                btn.html(originalText).prop('disabled', false);
            }
        },
        error: function(xhr, status, error) {
            console.error('Registration error:', error);
            showAlert('error', 'Registration failed. Please try again.');
            btn.html(originalText).prop('disabled', false);
        }
    });
}

// Submit Login
function submitLogin(email, password) {
    const btn = $('#login-form button[type="submit"]');
    const originalText = btn.html();

    btn.prop('disabled', true)
        .html('<i class="fas fa-spinner fa-spin me-2"></i> Logging in...');

    $.ajax({
        url: SITE_URL + 'api/v1/user/login',
        type: 'POST',
        data: JSON.stringify({
            email: email,
            password: password
        }),
        contentType: 'application/json',
        success: function(response) {
            if (response.success) {
                // Store token if provided
                if (response.data && response.data.token) {
                    localStorage.setItem('auth_token', response.data.token);
                }

                showAlert('success', 'Login successful! Redirecting...');
                setTimeout(() => {
                    window.location.href = SITE_URL + 'dashboard.php';
                }, 1000);
            } else {
                showAlert('error', response.error || 'Invalid credentials');
                btn.html(originalText).prop('disabled', false);
            }
        },
        error: function(xhr, status, error) {
            console.error('Login error:', error);
            showAlert('error', 'Login failed. Please try again.');
            btn.html(originalText).prop('disabled', false);
        }
    });
}

// Submit Forgot Password
function submitForgotPassword(email) {
    const btn = $('#forgot-password-form button[type="submit"]');
    const originalText = btn.html();

    btn.prop('disabled', true)
        .html('<i class="fas fa-spinner fa-spin me-2"></i> Sending...');

    $.ajax({
        url: SITE_URL + 'api/v1/user/forgot-password',
        type: 'POST',
        data: JSON.stringify({ email: email }),
        contentType: 'application/json',
        success: function(response) {
            if (response.success) {
                showAlert('success', 'Password reset instructions sent to your email');
                $('#forgot-password-form')[0].reset();
            } else {
                showAlert('error', response.error || 'Failed to send reset email');
            }
        },
        error: function() {
            showAlert('error', 'Failed to process request');
        },
        complete: function() {
            btn.html(originalText).prop('disabled', false);
        }
    });
}

// Generate QR Code
function generateQRCode() {
    const content = getQRContent();
    if (!content || content.trim() === '') {
        showAlert('error', 'Please enter content for your QR code');
        return;
    }

    const type = $('.qr-type-tab.active').data('type');
    const title = $('#qr-title').val() || 'Untitled QR Code';
    const isDynamic = $('#dynamic-qr').is(':checked');

    const qrData = {
        type: type,
        content: content,
        title: title,
        is_dynamic: isDynamic,
        style: {
            foreground_color: $('#fg-color').val(),
            background_color: $('#bg-color').val(),
            size: parseInt($('#qr-size').val()),
            margin: parseInt($('#qr-margin').val()),
            corner_style: $('#corner-style').val(),
            error_correction: $('#error-correction').val(),
            logo_path: $('#logo-path').val() || null
        }
    };

    // Show loading
    const btn = $('button[onclick="generateQRCode()"]');
    const originalText = btn.html();
    btn.html('<i class="fas fa-spinner fa-spin me-2"></i> Generating...');
    btn.prop('disabled', true);

    // Send to API
    $.ajax({
        url: SITE_URL + 'api/v1/qr/create',
        type: 'POST',
        data: JSON.stringify(qrData),
        contentType: 'application/json',
        headers: getAuthHeaders(),
        success: function(response) {
            if (response.success) {
                showAlert('success', 'QR code generated successfully!');

                // Show download options
                showDownloadOptions(response.data);

                // Update preview
                $('#qr-code-display').html(`
                    <img src="${response.data.image_url}" alt="Generated QR Code" 
                         class="img-fluid rounded shadow-sm">
                `);

                // Update QR ID for future operations
                if (response.data.id) {
                    $('#qr-id').val(response.data.id);
                }
            } else {
                showAlert('error', response.error || 'Failed to generate QR code');
            }
        },
        error: function(xhr) {
            if (xhr.status === 401) {
                showAlert('error', 'Please login to generate QR codes');
                setTimeout(() => {
                    window.location.href = SITE_URL + 'login.php';
                }, 2000);
            } else {
                showAlert('error', 'Failed to generate QR code. Please try again.');
            }
        },
        complete: function() {
            btn.html(originalText);
            btn.prop('disabled', false);
        }
    });
}

// Show Download Options Modal
function showDownloadOptions(data) {
    // Check if modal already exists
    if ($('#downloadModal').length) {
        $('#downloadModal').remove();
    }

    const modalHtml = `
        <div class="modal fade" id="downloadModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Download QR Code</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body text-center">
                        <div class="mb-4">
                            <img src="${data.image_url}" alt="QR Code" class="img-fluid rounded shadow" style="max-width: 200px;">
                        </div>
                        <p class="mb-3">Your QR code has been generated successfully!</p>
                        <div class="d-grid gap-2">
                            <button onclick="downloadQR('png')" class="btn btn-success">
                                <i class="fas fa-download me-2"></i> Download PNG
                            </button>
                            <button onclick="downloadQR('svg')" class="btn btn-info">
                                <i class="fas fa-download me-2"></i> Download SVG
                            </button>
                            <button onclick="downloadQR('pdf')" class="btn btn-warning">
                                <i class="fas fa-download me-2"></i> Download PDF
                            </button>
                            ${data.edit_url ? `
                            <a href="${data.edit_url}" class="btn btn-primary">
                                <i class="fas fa-edit me-2"></i> Edit QR Code
                            </a>
                            ` : ''}
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>
    `;

    $('body').append(modalHtml);
    const modalElement = document.getElementById('downloadModal');
    const modal = new bootstrap.Modal(modalElement);

    modal.show();

    // Remove modal on close
    modalElement.addEventListener('hidden.bs.modal', function() {
        $(this).remove();
    });
}

// Download QR Code
function downloadQR(format) {
    const qrId = $('#qr-id').val();
    const content = getQRContent();

    if (!content || content.trim() === '') {
        showAlert('error', 'Please generate a QR code first');
        return;
    }

    // If we have a QR ID, use it for download
    if (qrId) {
        window.open(SITE_URL + `api/v1/qr/download/${qrId}?format=${format}`, '_blank');
        return;
    }

    // Fallback to custom generation
    const qrData = {
        content: content,
        type: $('.qr-type-tab.active').data('type'),
        style: {
            foreground_color: $('#fg-color').val(),
            background_color: $('#bg-color').val(),
            size: parseInt($('#qr-size').val()),
            margin: parseInt($('#qr-margin').val()),
            corner_style: $('#corner-style').val(),
            error_correction: $('#error-correction').val(),
            logo_path: $('#logo-path').val() || null
        }
    };

    // Show loading
    showAlert('info', 'Preparing download...');

    // Call backend API to generate and download
    $.ajax({
        url: SITE_URL + 'api/v1/qr/download-custom',
        type: 'POST',
        data: JSON.stringify(qrData),
        contentType: 'application/json',
        xhrFields: {
            responseType: 'blob'
        },
        beforeSend: function(xhr) {
            xhr.setRequestHeader('Accept', `image/${format === 'pdf' ? 'pdf' : format}`);
        },
        success: function(blob, status, xhr) {
            // Get filename from header or generate one
            const filename = getFilenameFromHeaders(xhr) || `qrcode_${Date.now()}.${format}`;

            // Create download link
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();

            // Cleanup
            window.URL.revokeObjectURL(url);
            document.body.removeChild(a);

            showAlert('success', 'Download started!');
        },
        error: function(xhr, status, error) {
            console.error('Download error:', error);

            if (xhr.status === 401) {
                showAlert('error', 'Please login to download QR codes');
            } else {
                // Fallback to external service
                downloadViaExternalService(qrData, format);
            }
        }
    });
}

// Helper function to get filename from headers
function getFilenameFromHeaders(xhr) {
    const contentDisposition = xhr.getResponseHeader('Content-Disposition');
    if (contentDisposition) {
        const match = contentDisposition.match(/filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/);
        if (match && match[1]) {
            return match[1].replace(/['"]/g, '');
        }
    }
    return null;
}

// Fallback to external QR code service
function downloadViaExternalService(qrData, format) {
    const size = qrData.style.size || 300;
    const fgColor = qrData.style.foreground_color.replace('#', '') || '000000';
    const bgColor = qrData.style.background_color.replace('#', '') || 'ffffff';
    const margin = qrData.style.margin || 4;

    // Use external QR code API as fallback
    const downloadUrl = `https://qr.wala.guru/v1/create-qr-code/?
        size=${size}x${size}&
        data=${encodeURIComponent(qrData.content)}&
        color=${fgColor}&
        bgcolor=${bgColor}&
        format=${format}&
        margin=${margin}`.replace(/\s+/g, '');

    const a = document.createElement('a');
    a.href = downloadUrl;
    a.download = `qrcode.${format}`;
    a.target = '_blank';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);

    showAlert('info', 'Downloading via external service...');
}

// Save QR Code to Account
// function saveToAccount() {
//     if (!isLoggedIn()) {
//         showAlert('error', 'Please login to save QR codes');
//         setTimeout(() => {
//             window.location.href = SITE_URL + 'login.php';
//         }, 1500);
//         return;
//     }
//
//     const content = getQRContent();
//     if (!content || content.trim() === '') {
//         showAlert('error', 'Please generate a QR code first');
//         return;
//     }
//
//     const type = $('.qr-type-tab.active').data('type');
//     const title = $('#qr-title').val() || 'Untitled QR Code';
//     const isDynamic = $('#dynamic-qr').is(':checked');
//
//     const qrData = {
//         type: type,
//         content: content,
//         title: title,
//         is_dynamic: isDynamic,
//         style: {
//             foreground_color: $('#fg-color').val(),
//             background_color: $('#bg-color').val(),
//             size: parseInt($('#qr-size').val()),
//             margin: parseInt($('#qr-margin').val()),
//             corner_style: $('#corner-style').val(),
//             error_correction: $('#error-correction').val(),
//             logo_path: $('#logo-path').val() || null
//         }
//     };
//
//     const btn = $('button[onclick="saveToAccount()"]');
//     const originalText = btn.html();
//     btn.html('<i class="fas fa-spinner fa-spin me-2"></i> Saving...');
//     btn.prop('disabled', true);
//
//     $.ajax({
//         url: SITE_URL + 'api/v1/qr/save',
//         type: 'POST',
//         data: JSON.stringify(qrData),
//         contentType: 'application/json',
//         headers: getAuthHeaders(),
//         success: function(response) {
//             if (response.success) {
//                 showAlert('success', 'QR code saved to your account');
//                 if (response.data.id) {
//                     $('#qr-id').val(response.data.id);
//                 }
//             } else {
//                 showAlert('error', response.error || 'Failed to save QR code');
//             }
//         },
//         error: function() {
//             showAlert('error', 'Failed to save QR code');
//         },
//         complete: function() {
//             btn.html(originalText);
//             btn.prop('disabled', false);
//         }
//     });
// }

// Get Share Link
function getShareLink() {
    const qrId = $('#qr-id').val();

    if (!qrId) {
        showAlert('error', 'Please save or generate a QR code first');
        return;
    }

    const shareUrl = SITE_URL + 'share/' + qrId;

    // Copy to clipboard
    copyToClipboard(shareUrl);

    // Show share options
    showAlert('success', `Share link copied to clipboard!<br><small>${shareUrl}</small>`);
}

// Show Alert
function showAlert(type, message) {
    // Remove existing alerts
    $('.alert-dismissible.position-fixed').remove();

    const alertHtml = `
        <div class="alert alert-${type} alert-dismissible fade show position-fixed" 
             style="top: 20px; right: 20px; z-index: 1060; min-width: 300px; max-width: 400px;">
            <div class="d-flex align-items-start">
                <div class="flex-grow-1">
                    ${message}
                </div>
                <button type="button" class="btn-close ms-2" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        </div>
    `;

    $('body').append(alertHtml);

    // Auto remove after 5 seconds
    setTimeout(() => {
        const alert = $('.alert-dismissible.position-fixed');
        if (alert.length) {
            alert.alert('close');
        }
    }, 5000);
}

// Get Authentication Headers
function getAuthHeaders() {
    const headers = {
        'Content-Type': 'application/json'
    };

    // Check for API token
    const token = localStorage.getItem('auth_token');
    if (token) {
        headers['Authorization'] = `Bearer ${token}`;
    }

    // Check for session
    if (isLoggedIn()) {
        // Session cookie will be sent automatically
    }

    return headers;
}

// Check if User is Logged In
function isLoggedIn() {
    // Check global variable
    if (typeof USER_LOGGED_IN !== 'undefined') {
        return USER_LOGGED_IN === 'true';
    }

    // Check local storage
    return localStorage.getItem('auth_token') !== null;
}

// Initialize Charts
function initializeCharts() {
    if (typeof Chart === 'undefined') {
        console.error('Chart.js library not loaded');
        return;
    }

    try {
        // Analytics chart
        if ($('#analyticsChart').length) {
            loadAnalyticsChart();
        }

        // User stats chart
        if ($('#userStatsChart').length) {
            loadUserStatsChart();
        }
    } catch (error) {
        console.error('Error initializing charts:', error);
    }
}

// Load Analytics Chart
function loadAnalyticsChart() {
    if (!isLoggedIn()) {
        return;
    }

    $.ajax({
        url: SITE_URL + 'api/v1/analytics/user',
        headers: getAuthHeaders(),
        success: function(response) {
            if (response.success && response.data) {
                renderAnalyticsChart(response.data);
            }
        },
        error: function() {
            console.error('Failed to load analytics data');
        }
    });
}

// Render Analytics Chart
function renderAnalyticsChart(data) {
    const ctx = document.getElementById('analyticsChart').getContext('2d');

    // Destroy existing chart if exists
    if (window.analyticsChart instanceof Chart) {
        window.analyticsChart.destroy();
    }

    window.analyticsChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: data.labels || [],
            datasets: [{
                label: 'Scans',
                data: data.values || [],
                borderColor: '#4a6fa5',
                backgroundColor: 'rgba(74, 111, 165, 0.1)',
                borderWidth: 2,
                fill: true,
                tension: 0.4,
                pointBackgroundColor: '#4a6fa5',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    mode: 'index',
                    intersect: false,
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    padding: 12
                }
            },
            scales: {
                x: {
                    grid: {
                        display: false
                    }
                },
                y: {
                    beginAtZero: true,
                    ticks: {
                        precision: 0
                    },
                    grid: {
                        color: 'rgba(0, 0, 0, 0.05)'
                    }
                }
            }
        }
    });
}

// Load User Stats Chart
function loadUserStatsChart() {
    if (!isLoggedIn()) {
        return;
    }

    $.ajax({
        url: SITE_URL + 'api/v1/user/stats',
        headers: getAuthHeaders(),
        success: function(response) {
            if (response.success && response.data) {
                renderUserStatsChart(response.data);
            }
        },
        error: function() {
            console.error('Failed to load user stats');
        }
    });
}

// Render User Stats Chart
function renderUserStatsChart(data) {
    const ctx = document.getElementById('userStatsChart').getContext('2d');

    // Destroy existing chart if exists
    if (window.userStatsChart instanceof Chart) {
        window.userStatsChart.destroy();
    }

    window.userStatsChart = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: ['QR Codes', 'Total Scans', 'Active Days'],
            datasets: [{
                data: [
                    data.total_qr_codes || 0,
                    data.total_scans || 0,
                    data.active_days || 0
                ],
                backgroundColor: ['#4a6fa5', '#4fc3a1', '#ff6b6b'],
                borderWidth: 2,
                borderColor: '#ffffff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 20,
                        usePointStyle: true
                    }
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `${context.label}: ${context.raw}`;
                        }
                    }
                }
            },
            cutout: '70%'
        }
    });
}

// Copy to Clipboard
function copyToClipboard(text) {
    if (!text) {
        showAlert('error', 'Nothing to copy');
        return;
    }

    if (!navigator.clipboard) {
        // Fallback for older browsers
        const textarea = document.createElement('textarea');
        textarea.value = text;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        showAlert('success', 'Copied to clipboard!');
        return;
    }

    navigator.clipboard.writeText(text).then(function() {
        showAlert('success', 'Copied to clipboard!');
    }, function(err) {
        console.error('Copy failed:', err);
        showAlert('error', 'Failed to copy');
    });
}

// Export Data
function exportData(type, data) {
    if (!data || data.length === 0) {
        showAlert('error', 'No data to export');
        return;
    }

    let content, filename, mimeType;

    switch(type) {
        case 'csv':
            content = convertToCSV(data);
            filename = `export_${formatDate(new Date(), 'YYYY-MM-DD')}.csv`;
            mimeType = 'text/csv;charset=utf-8;';
            break;
        case 'json':
            content = JSON.stringify(data, null, 2);
            filename = `export_${formatDate(new Date(), 'YYYY-MM-DD')}.json`;
            mimeType = 'application/json';
            break;
        case 'excel':
            // For Excel export, we would need a library like SheetJS
            showAlert('info', 'Excel export coming soon!');
            return;
        default:
            showAlert('error', 'Unsupported export format');
            return;
    }

    const blob = new Blob([content], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    showAlert('success', `${type.toUpperCase()} file downloaded successfully!`);
}

// Convert to CSV
function convertToCSV(data) {
    if (!Array.isArray(data) || data.length === 0) return '';

    const headers = Object.keys(data[0]);
    const csvRows = [];

    // Add header row
    csvRows.push(headers.join(','));

    // Add data rows
    data.forEach(row => {
        const values = headers.map(header => {
            const value = row[header];
            if (value === null || value === undefined) return '';

            // Escape quotes and wrap in quotes if contains comma or quotes
            let escaped = String(value).replace(/"/g, '""');
            if (escaped.includes(',') || escaped.includes('"') || escaped.includes('\n')) {
                escaped = `"${escaped}"`;
            }
            return escaped;
        });
        csvRows.push(values.join(','));
    });

    return csvRows.join('\n');
}

// Format Date
function formatDate(date, format = 'YYYY-MM-DD') {
    if (!(date instanceof Date)) {
        date = new Date(date);
    }

    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    const seconds = String(date.getSeconds()).padStart(2, '0');

    return format
        .replace('YYYY', year)
        .replace('MM', month)
        .replace('DD', day)
        .replace('HH', hours)
        .replace('mm', minutes)
        .replace('ss', seconds);
}

// Format Number
function formatNumber(num) {
    if (typeof num !== 'number') {
        num = parseFloat(num) || 0;
    }

    if (num >= 1000000) {
        return (num / 1000000).toFixed(1).replace(/\.0$/, '') + 'M';
    }
    if (num >= 1000) {
        return (num / 1000).toFixed(1).replace(/\.0$/, '') + 'K';
    }
    return num.toString();
}

// Debounce function
function debounce(func, wait, immediate) {
    let timeout;
    return function() {
        const context = this;
        const args = arguments;
        const later = function() {
            timeout = null;
            if (!immediate) func.apply(context, args);
        };
        const callNow = immediate && !timeout;
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
        if (callNow) func.apply(context, args);
    };
}

// Throttle function
function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// Logout function
function logout() {
    $.ajax({
        url: SITE_URL + 'api/v1/user/logout',
        type: 'POST',
        headers: getAuthHeaders(),
        success: function(response) {
            // Clear local storage
            localStorage.removeItem('auth_token');
            localStorage.removeItem('qr_api_key');

            // Redirect to login
            window.location.href = SITE_URL + 'login.php';
        },
        error: function() {
            // Still redirect even if API fails
            localStorage.clear();
            window.location.href = SITE_URL + 'login.php';
        }
    });
}

// Toggle QR Type Tab
function toggleQrType(type) {
    // Update active tab
    $('.qr-type-tab').removeClass('active');
    $(`.qr-type-tab[data-type="${type}"]`).addClass('active');

    // Show corresponding content
    $('.qr-type-content').hide();
    $(`#${type}-content`).show();

    // Update preview
    updateQRPreview();
}

// Validate URL
function isValidUrl(string) {
    try {
        new URL(string);
        return true;
    } catch (_) {
        return false;
    }
}

// Generate Random QR ID
function generateQrId() {
    return 'qr_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
}

// Global Variables
const SITE_URL = window.SITE_URL || 'https://qr.wala.guru/';
const USER_LOGGED_IN = typeof window.USER_LOGGED_IN !== 'undefined' ? window.USER_LOGGED_IN : false;

// Error handler for global errors
window.addEventListener('error', function(e) {
    console.error('Global error:', e.error);
    showAlert('error', 'An error occurred. Please refresh the page.');
});

// Unhandled promise rejection handler
window.addEventListener('unhandledrejection', function(e) {
    console.error('Unhandled promise rejection:', e.reason);
    showAlert('error', 'An error occurred. Please try again.');
});

// Make functions available globally
window.copyToClipboard = copyToClipboard;
window.downloadQR = downloadQR;
window.generateQRCode = generateQRCode;
window.saveToAccount = saveToAccount;
window.getShareLink = getShareLink;
window.toggleQrType = toggleQrType;
window.logout = logout;