<?php
require_once __DIR__ . '/../config/database.php';

class User {
    private $db;
    private $table = 'users';

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Create new user
    public function create($data) {
        $sql = "INSERT INTO {$this->table} 
                (email, password_hash, full_name, api_key, subscription_type) 
                VALUES (:email, :password_hash, :full_name, :api_key, :subscription_type)";

        $api_key = bin2hex(random_bytes(32));
        $data['password_hash'] = password_hash($data['password'], PASSWORD_DEFAULT);

        $params = [
            ':email' => $data['email'],
            ':password_hash' => $data['password_hash'],
            ':full_name' => $data['full_name'] ?? null,
            ':api_key' => $api_key,
            ':subscription_type' => $data['subscription_type'] ?? 'free'
        ];

        $stmt = Database::getInstance()->execute($sql, $params);

        if ($stmt) {
            return [
                'id' => Database::getInstance()->lastInsertId(),
                'api_key' => $api_key
            ];
        }
        return false;
    }

    // Find user by email
    public function findByEmail($email) {
        $sql = "SELECT * FROM {$this->table} WHERE email = :email";
        $stmt = Database::getInstance()->execute($sql, [':email' => $email]);
        return $stmt->fetch();
    }

    // Find user by ID
    public function findById($id) {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id";
        $stmt = Database::getInstance()->execute($sql, [':id' => $id]);
        return $stmt->fetch();
    }

    // Find user by API key
    public function findByApiKey($api_key) {
        $sql = "SELECT * FROM {$this->table} WHERE api_key = :api_key";
        $stmt = Database::getInstance()->execute($sql, [':api_key' => $api_key]);
        return $stmt->fetch();
    }

    // Authenticate user
    public function authenticate($email, $password) {
        $user = $this->findByEmail($email);

        if ($user && password_verify($password, $user['password_hash'])) {
            // Update last login
            $this->updateLastLogin($user['id']);
            return $user;
        }
        return false;
    }

    // Update user
    public function update($id, $data) {
        $fields = [];
        $params = [':id' => $id];

        foreach ($data as $key => $value) {
            if ($key === 'password') {
                $fields[] = "password_hash = :password_hash";
                $params[':password_hash'] = password_hash($value, PASSWORD_DEFAULT);
            } else {
                $fields[] = "{$key} = :{$key}";
                $params[":{$key}"] = $value;
            }
        }

        $sql = "UPDATE {$this->table} SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = :id";
        return Database::getInstance()->execute($sql, $params);
    }

    // Update last login
    private function updateLastLogin($id) {
        $sql = "UPDATE {$this->table} SET last_active = NOW() WHERE id = :id";
        return Database::getInstance()->execute($sql, [':id' => $id]);
    }

    // Check if user can create more QR codes
    public function canCreateQRCode($user_id) {
        $sql = "SELECT 
                    u.subscription_type,
                    u.qr_generated,
                    u.qr_limit,
                    COUNT(q.id) as current_count
                FROM {$this->table} u
                LEFT JOIN qr_codes q ON u.id = q.user_id
                WHERE u.id = :user_id
                GROUP BY u.id";

        $stmt = Database::getInstance()->execute($sql, [':user_id' => $user_id]);
        $user = $stmt->fetch();

        if (!$user) return false;

        // Check limit
        $limit = $user['subscription_type'] === 'premium' ? MAX_QR_PER_USER_PREMIUM : MAX_QR_PER_USER_FREE;

        if ($user['current_count'] >= $limit) {
            return false;
        }

        // Update generated count
        $sql = "UPDATE {$this->table} SET qr_generated = qr_generated + 1 WHERE id = :user_id";
        Database::getInstance()->execute($sql, [':user_id' => $user_id]);

        return true;
    }

    // Get user stats
    public function getStats($user_id) {
        $sql = "SELECT 
                    u.email,
                    u.full_name,
                    u.subscription_type,
                    u.created_at,
                    u.last_active,
                    COUNT(q.id) as total_qr_codes,
                    COALESCE(SUM(q.scan_count), 0) as total_scans
                FROM {$this->table} u
                LEFT JOIN qr_codes q ON u.id = q.user_id
                WHERE u.id = :user_id";

        $stmt = Database::getInstance()->execute($sql, [':user_id' => $user_id]);
        return $stmt->fetch();
    }
}
?>