<?php
require_once __DIR__ . '/../config/database.php';

class QrCode {
    private $db;
    private $table = 'qr_codes';

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Create QR code
    public function create($data) {
        $public_id = $this->generatePublicId();

        $sql = "INSERT INTO {$this->table} 
                (public_id, user_id, qr_type, content, title, settings, is_dynamic, expires_at) 
                VALUES (:public_id, :user_id, :qr_type, :content, :title, :settings, :is_dynamic, :expires_at)";

        $params = [
            ':public_id' => $public_id,
            ':user_id' => $data['user_id'] ?? null,
            ':qr_type' => $data['qr_type'],
            ':content' => $data['content'],
            ':title' => $data['title'] ?? null,
            ':settings' => json_encode($data['settings'] ?? []),
            ':is_dynamic' => $data['is_dynamic'] ? 1 : 0,
            ':expires_at' => $data['expires_at'] ?? null
        ];

        $stmt = Database::getInstance()->execute($sql, $params);

        if ($stmt) {
            return [
                'id' => Database::getInstance()->lastInsertId(),
                'public_id' => $public_id
            ];
        }
        return false;
    }

    // Get QR code by public ID
    public function findByPublicId($public_id) {
        $sql = "SELECT * FROM {$this->table} WHERE public_id = :public_id";
        $stmt = Database::getInstance()->execute($sql, [':public_id' => $public_id]);
        return $stmt->fetch();
    }

    // Get QR code by ID
    public function findById($id) {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id";
        $stmt = Database::getInstance()->execute($sql, [':id' => $id]);
        return $stmt->fetch();
    }

    // Get user's QR codes
    public function getUserQRCodes($user_id, $limit = 50, $offset = 0) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE user_id = :user_id 
                ORDER BY created_at DESC 
                LIMIT :limit OFFSET :offset";

        $params = [
            ':user_id' => $user_id,
            ':limit' => (int)$limit,
            ':offset' => (int)$offset
        ];

        $stmt = Database::getInstance()->execute($sql, $params);
        return $stmt->fetchAll();
    }

    // Update QR code
    public function update($id, $data) {
        $fields = [];
        $params = [':id' => $id];

        foreach ($data as $key => $value) {
            if ($key === 'settings') {
                $fields[] = "settings = :settings";
                $params[':settings'] = json_encode($value);
            } else {
                $fields[] = "{$key} = :{$key}";
                $params[":{$key}"] = $value;
            }
        }

        $sql = "UPDATE {$this->table} SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = :id";
        return Database::getInstance()->execute($sql, $params);
    }

    // Delete QR code
    public function delete($id) {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        return Database::getInstance()->execute($sql, [':id' => $id]);
    }

    // Increment scan count
    public function incrementScanCount($id) {
        $sql = "UPDATE {$this->table} SET scan_count = scan_count + 1 WHERE id = :id";
        return Database::getInstance()->execute($sql, [':id' => $id]);
    }

    // Search QR codes
    public function search($user_id, $query, $limit = 50, $offset = 0) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE user_id = :user_id 
                AND (title LIKE :query OR content LIKE :query)
                ORDER BY created_at DESC 
                LIMIT :limit OFFSET :offset";

        $params = [
            ':user_id' => $user_id,
            ':query' => "%{$query}%",
            ':limit' => (int)$limit,
            ':offset' => (int)$offset
        ];

        $stmt = Database::getInstance()->execute($sql, $params);
        return $stmt->fetchAll();
    }

    // Generate unique public ID
    private function generatePublicId() {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $length = 8;

        do {
            $public_id = '';
            for ($i = 0; $i < $length; $i++) {
                $public_id .= $characters[rand(0, strlen($characters) - 1)];
            }
        } while ($this->publicIdExists($public_id));

        return $public_id;
    }

    // Check if public ID exists
    private function publicIdExists($public_id) {
        $sql = "SELECT id FROM {$this->table} WHERE public_id = :public_id LIMIT 1";
        $stmt = Database::getInstance()->execute($sql, [':public_id' => $public_id]);
        return $stmt->rowCount() > 0;
    }

    // Get QR code styles
    public function getStyles($qr_code_id) {
        $sql = "SELECT * FROM qr_styles WHERE qr_code_id = :qr_code_id";
        $stmt = Database::getInstance()->execute($sql, [':qr_code_id' => $qr_code_id]);
        return $stmt->fetch();
    }

    // Save QR code styles
    public function saveStyles($qr_code_id, $styles) {
        // Check if style exists
        $existing = $this->getStyles($qr_code_id);

        if ($existing) {
            // Update existing
            $sql = "UPDATE qr_styles SET 
                    foreground_color = :foreground_color,
                    background_color = :background_color,
                    logo_path = :logo_path,
                    gradient_type = :gradient_type,
                    gradient_colors = :gradient_colors,
                    corner_style = :corner_style,
                    frame_text = :frame_text,
                    frame_style = :frame_style,
                    size = :size,
                    margin = :margin,
                    error_correction = :error_correction
                    WHERE qr_code_id = :qr_code_id";
        } else {
            // Insert new
            $sql = "INSERT INTO qr_styles 
                    (qr_code_id, foreground_color, background_color, logo_path, gradient_type, 
                     gradient_colors, corner_style, frame_text, frame_style, size, margin, error_correction) 
                    VALUES (:qr_code_id, :foreground_color, :background_color, :logo_path, :gradient_type, 
                    :gradient_colors, :corner_style, :frame_text, :frame_style, :size, :margin, :error_correction)";
        }

        $params = [
            ':qr_code_id' => $qr_code_id,
            ':foreground_color' => $styles['foreground_color'] ?? QR_DEFAULT_COLOR_FG,
            ':background_color' => $styles['background_color'] ?? QR_DEFAULT_COLOR_BG,
            ':logo_path' => $styles['logo_path'] ?? null,
            ':gradient_type' => $styles['gradient_type'] ?? 'none',
            ':gradient_colors' => json_encode($styles['gradient_colors'] ?? []),
            ':corner_style' => $styles['corner_style'] ?? 'square',
            ':frame_text' => $styles['frame_text'] ?? null,
            ':frame_style' => $styles['frame_style'] ?? 'none',
            ':size' => $styles['size'] ?? QR_DEFAULT_SIZE,
            ':margin' => $styles['margin'] ?? QR_DEFAULT_MARGIN,
            ':error_correction' => $styles['error_correction'] ?? QR_DEFAULT_ERROR_CORRECTION
        ];

        return Database::getInstance()->execute($sql, $params);
    }
}
?>