<?php
require_once __DIR__ . '/../config/database.php';

class Analytics {
    private $db;
    private $table = 'qr_analytics';

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Log scan
    public function logScan($qr_code_id, $data = []) {
        $sql = "INSERT INTO {$this->table} 
                (qr_code_id, scan_date, scan_time, ip_address, user_agent, 
                 country, city, device_type, browser, os, referrer) 
                VALUES (:qr_code_id, :scan_date, :scan_time, :ip_address, :user_agent, 
                        :country, :city, :device_type, :browser, :os, :referrer)";

        $params = [
            ':qr_code_id' => $qr_code_id,
            ':scan_date' => date('Y-m-d'),
            ':scan_time' => date('H:i:s'),
            ':ip_address' => $data['ip_address'] ?? $_SERVER['REMOTE_ADDR'] ?? null,
            ':user_agent' => $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? null,
            ':country' => $data['country'] ?? null,
            ':city' => $data['city'] ?? null,
            ':device_type' => $data['device_type'] ?? $this->detectDeviceType(),
            ':browser' => $data['browser'] ?? $this->detectBrowser(),
            ':os' => $data['os'] ?? $this->detectOS(),
            ':referrer' => $data['referrer'] ?? $_SERVER['HTTP_REFERER'] ?? null
        ];

        return Database::getInstance()->execute($sql, $params);
    }

    // Get analytics for QR code
    public function getQRCodeAnalytics($qr_code_id, $start_date = null, $end_date = null) {
        $where = "WHERE qr_code_id = :qr_code_id";
        $params = [':qr_code_id' => $qr_code_id];

        if ($start_date) {
            $where .= " AND scan_date >= :start_date";
            $params[':start_date'] = $start_date;
        }

        if ($end_date) {
            $where .= " AND scan_date <= :end_date";
            $params[':end_date'] = $end_date;
        }

        // Get daily scans
        $sql_daily = "SELECT scan_date, COUNT(*) as scans 
                      FROM {$this->table} 
                      {$where} 
                      GROUP BY scan_date 
                      ORDER BY scan_date";

        // Get device distribution
        $sql_devices = "SELECT device_type, COUNT(*) as scans 
                        FROM {$this->table} 
                        {$where} 
                        GROUP BY device_type";

        // Get browser distribution
        $sql_browsers = "SELECT browser, COUNT(*) as scans 
                         FROM {$this->table} 
                         {$where} AND browser IS NOT NULL 
                         GROUP BY browser 
                         ORDER BY scans DESC 
                         LIMIT 10";

        // Get country distribution
        $sql_countries = "SELECT country, COUNT(*) as scans 
                          FROM {$this->table} 
                          {$where} AND country IS NOT NULL 
                          GROUP BY country 
                          ORDER BY scans DESC 
                          LIMIT 10";

        $daily = Database::getInstance()->execute($sql_daily, $params)->fetchAll();
        $devices = Database::getInstance()->execute($sql_devices, $params)->fetchAll();
        $browsers = Database::getInstance()->execute($sql_browsers, $params)->fetchAll();
        $countries = Database::getInstance()->execute($sql_countries, $params)->fetchAll();

        return [
            'daily' => $daily,
            'devices' => $devices,
            'browsers' => $browsers,
            'countries' => $countries,
            'total' => array_sum(array_column($daily, 'scans'))
        ];
    }

    // Get user analytics
    public function getUserAnalytics($user_id, $start_date = null, $end_date = null) {
        $where = "WHERE q.user_id = :user_id";
        $params = [':user_id' => $user_id];

        if ($start_date) {
            $where .= " AND a.scan_date >= :start_date";
            $params[':start_date'] = $start_date;
        }

        if ($end_date) {
            $where .= " AND a.scan_date <= :end_date";
            $params[':end_date'] = $end_date;
        }

        $sql = "SELECT 
                    q.title as qr_title,
                    q.public_id,
                    COUNT(a.id) as total_scans,
                    COUNT(DISTINCT DATE(a.scan_date)) as active_days,
                    MIN(a.scan_date) as first_scan,
                    MAX(a.scan_date) as last_scan
                FROM qr_codes q
                LEFT JOIN {$this->table} a ON q.id = a.qr_code_id
                {$where}
                GROUP BY q.id
                ORDER BY total_scans DESC";

        $stmt = Database::getInstance()->execute($sql, $params);
        return $stmt->fetchAll();
    }

    // Detect device type
    private function detectDeviceType() {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        if (preg_match('/(mobile|android|iphone|ipod|blackberry|opera mini|windows phone)/i', $user_agent)) {
            return 'mobile';
        } elseif (preg_match('/(tablet|ipad|playbook|kindle)/i', $user_agent)) {
            return 'tablet';
        } else {
            return 'desktop';
        }
    }

    // Detect browser
    private function detectBrowser() {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        if (strpos($user_agent, 'Firefox') !== false) return 'Firefox';
        if (strpos($user_agent, 'Chrome') !== false) return 'Chrome';
        if (strpos($user_agent, 'Safari') !== false) return 'Safari';
        if (strpos($user_agent, 'Edge') !== false) return 'Edge';
        if (strpos($user_agent, 'MSIE') !== false || strpos($user_agent, 'Trident') !== false) return 'IE';
        if (strpos($user_agent, 'Opera') !== false) return 'Opera';

        return 'Unknown';
    }

    // Detect OS
    private function detectOS() {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        if (strpos($user_agent, 'Windows') !== false) return 'Windows';
        if (strpos($user_agent, 'Mac') !== false) return 'MacOS';
        if (strpos($user_agent, 'Linux') !== false) return 'Linux';
        if (strpos($user_agent, 'Android') !== false) return 'Android';
        if (strpos($user_agent, 'iPhone') !== false || strpos($user_agent, 'iPad') !== false) return 'iOS';

        return 'Unknown';
    }
}
?>