<?php
require_once __DIR__ . '/../config/database.php';

class RateLimit {
    private $db;
    private $table = 'rate_limits';

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Check rate limit for IP address
    public function checkIPLimit($limit = 100, $window = 3600) {
        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        $now = time();

        // Clean up old entries
        $this->cleanup($window);

        // Check current count
        $sql = "SELECT COUNT(*) as count FROM {$this->table} 
                WHERE ip_address = :ip AND timestamp > :timestamp";

        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':ip' => $ip,
            ':timestamp' => $now - $window
        ]);

        $result = $stmt->fetch();
        $count = $result['count'] ?? 0;

        if ($count >= $limit) {
            return false;
        }

        // Log this request
        $sql = "INSERT INTO {$this->table} (ip_address, timestamp) VALUES (:ip, :timestamp)";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':ip' => $ip,
            ':timestamp' => $now
        ]);

        return true;
    }

    // Check rate limit for API key
    public function checkApiKeyLimit($api_key, $limit = 1000, $window = 3600) {
        if (empty($api_key)) {
            return $this->checkIPLimit($limit, $window);
        }

        $now = time();

        // Clean up old entries
        $this->cleanup($window);

        // Check current count
        $sql = "SELECT COUNT(*) as count FROM {$this->table} 
                WHERE api_key = :api_key AND timestamp > :timestamp";

        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':api_key' => $api_key,
            ':timestamp' => $now - $window
        ]);

        $result = $stmt->fetch();
        $count = $result['count'] ?? 0;

        if ($count >= $limit) {
            return false;
        }

        // Log this request
        $sql = "INSERT INTO {$this->table} (api_key, timestamp) VALUES (:api_key, :timestamp)";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':api_key' => $api_key,
            ':timestamp' => $now
        ]);

        return true;
    }

    // Check rate limit for user
    public function checkUserLimit($user_id, $limit = 5000, $window = 3600) {
        $now = time();

        // Clean up old entries
        $this->cleanup($window);

        // Check current count
        $sql = "SELECT COUNT(*) as count FROM {$this->table} 
                WHERE user_id = :user_id AND timestamp > :timestamp";

        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':user_id' => $user_id,
            ':timestamp' => $now - $window
        ]);

        $result = $stmt->fetch();
        $count = $result['count'] ?? 0;

        if ($count >= $limit) {
            return false;
        }

        // Log this request
        $sql = "INSERT INTO {$this->table} (user_id, timestamp) VALUES (:user_id, :timestamp)";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':user_id' => $user_id,
            ':timestamp' => $now
        ]);

        return true;
    }

    // Clean up old entries
    private function cleanup($window) {
        $sql = "DELETE FROM {$this->table} WHERE timestamp < :timestamp";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':timestamp' => time() - $window
        ]);
    }

    // Get rate limit headers
    public function getHeaders($identifier, $limit, $window) {
        $now = time();

        // Get count
        $sql = "SELECT COUNT(*) as count FROM {$this->table} 
                WHERE identifier = :identifier AND timestamp > :timestamp";

        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':identifier' => $identifier,
            ':timestamp' => $now - $window
        ]);

        $result = $stmt->fetch();
        $count = $result['count'] ?? 0;
        $remaining = max(0, $limit - $count);

        return [
            'X-RateLimit-Limit' => $limit,
            'X-RateLimit-Remaining' => $remaining,
            'X-RateLimit-Reset' => $now + $window
        ];
    }

    // Apply rate limiting middleware
    public static function apply($type = 'ip', $limit = 100, $window = 3600) {
        $rateLimiter = new self();
        $headers = getallheaders();

        switch ($type) {
            case 'api_key':
                $api_key = $headers['Authorization'] ?? $headers['authorization'] ?? '';
                $api_key = str_replace('Bearer ', '', $api_key);

                if (empty($api_key)) {
                    $api_key = $_GET['api_key'] ?? '';
                }

                if (!$rateLimiter->checkApiKeyLimit($api_key, $limit, $window)) {
                    http_response_code(429);
                    echo json_encode(['error' => 'Rate limit exceeded']);
                    exit();
                }
                break;

            case 'user':
                $auth = new Auth();
                $user = $auth->getCurrentUser();

                if ($user && !$rateLimiter->checkUserLimit($user['id'], $limit, $window)) {
                    http_response_code(429);
                    echo json_encode(['error' => 'Rate limit exceeded']);
                    exit();
                }
                break;

            case 'ip':
            default:
                if (!$rateLimiter->checkIPLimit($limit, $window)) {
                    http_response_code(429);
                    echo json_encode(['error' => 'Rate limit exceeded']);
                    exit();
                }
                break;
        }
    }
}
?>