<?php
require_once __DIR__ . '/../../vendor/autoload.php';

use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;
use chillerlan\QRCode\Output\QRImageOutput;

class QrGenerator {

    public function generate($data, $options = []) {
        // Default options
        $default_options = [
            'size' => QR_DEFAULT_SIZE,
            'margin' => QR_DEFAULT_MARGIN,
            'foreground_color' => QR_DEFAULT_COLOR_FG,
            'background_color' => QR_DEFAULT_COLOR_BG,
            'error_correction' => QR_DEFAULT_ERROR_CORRECTION,
            'logo_path' => null,
            'gradient_type' => 'none',
            'gradient_colors' => [],
            'corner_style' => 'square'
        ];

        $options = array_merge($default_options, $options);

        // Create QR code options
        $qr_options = new QROptions([
            'version' => 7,
            'outputType' => QRCode::OUTPUT_IMAGE_PNG,
            'eccLevel' => constant("chillerlan\\QRCode\\QRCode::ECC_" . $options['error_correction']),
            'scale' => 5,
            'imageBase64' => false,
            'imageTransparent' => false,
            'outputInterface' => QRCode::OUTPUT_IMAGE_PNG,
            'returnResource' => true,
        ]);

        // Generate QR code
        $qr_code = new QRCode($qr_options);
        $qr_image = $qr_code->render($data);

        // Apply styling
        $styled_image = $this->applyStyling($qr_image, $options);

        return $styled_image;
    }

    private function applyStyling($image_resource, $options) {
        // Apply colors
        $styled_image = $this->applyColors($image_resource, $options);

        // Apply corner style
        $styled_image = $this->applyCornerStyle($styled_image, $options['corner_style']);

        // Apply gradient if needed
        if ($options['gradient_type'] != 'none' && !empty($options['gradient_colors'])) {
            $styled_image = $this->applyGradient($styled_image, $options);
        }

        // Add logo if provided
        if ($options['logo_path'] && file_exists($options['logo_path'])) {
            $styled_image = $this->addLogo($styled_image, $options['logo_path'], $options['size']);
        }

        return $styled_image;
    }

    private function applyColors($image, $options) {
        // Create new image with specified colors
        $width = imagesx($image);
        $height = imagesy($image);

        $new_image = imagecreatetruecolor($width, $height);

        // Convert hex colors to RGB
        $fg_rgb = $this->hexToRgb($options['foreground_color']);
        $bg_rgb = $this->hexToRgb($options['background_color']);

        $fg_color = imagecolorallocate($new_image, $fg_rgb[0], $fg_rgb[1], $fg_rgb[2]);
        $bg_color = imagecolorallocate($new_image, $bg_rgb[0], $bg_rgb[1], $bg_rgb[2]);

        // Fill background
        imagefilledrectangle($new_image, 0, 0, $width, $height, $bg_color);

        // Copy QR code with foreground color
        for ($x = 0; $x < $width; $x++) {
            for ($y = 0; $y < $height; $y++) {
                $rgb = imagecolorat($image, $x, $y);
                $color = imagecolorsforindex($image, $rgb);

                // If pixel is black (QR code), use foreground color
                if ($color['red'] < 128 && $color['green'] < 128 && $color['blue'] < 128) {
                    imagesetpixel($new_image, $x, $y, $fg_color);
                }
            }
        }

        imagedestroy($image);
        return $new_image;
    }

    private function applyCornerStyle($image, $corner_style) {
        $width = imagesx($image);
        $height = imagesy($image);

        switch ($corner_style) {
            case 'rounded':
                return $this->roundCorners($image, 20);
            case 'circle':
                return $this->roundCorners($image, $width / 2);
            case 'dots':
                return $this->applyDotStyle($image);
            default:
                return $image;
        }
    }

    private function roundCorners($image, $radius) {
        $width = imagesx($image);
        $height = imagesy($image);

        $new_image = imagecreatetruecolor($width, $height);
        imagesavealpha($new_image, true);
        $transparent = imagecolorallocatealpha($new_image, 0, 0, 0, 127);
        imagefill($new_image, 0, 0, $transparent);

        // Create rounded rectangle
        $this->imageFilledRoundedRectangle($new_image, 0, 0, $width, $height, $radius, $transparent);

        // Copy original image
        imagecopy($new_image, $image, 0, 0, 0, 0, $width, $height);

        imagedestroy($image);
        return $new_image;
    }

    private function imageFilledRoundedRectangle($img, $x, $y, $x2, $y2, $radius, $color) {
        // This is a simplified implementation
        imagefilledrectangle($img, $x + $radius, $y, $x2 - $radius, $y2, $color);
        imagefilledrectangle($img, $x, $y + $radius, $x2, $y2 - $radius, $color);

        // Draw circles for corners
        imagefilledellipse($img, $x + $radius, $y + $radius, $radius * 2, $radius * 2, $color);
        imagefilledellipse($img, $x2 - $radius, $y + $radius, $radius * 2, $radius * 2, $color);
        imagefilledellipse($img, $x + $radius, $y2 - $radius, $radius * 2, $radius * 2, $color);
        imagefilledellipse($img, $x2 - $radius, $y2 - $radius, $radius * 2, $radius * 2, $color);
    }

    private function applyDotStyle($image) {
        // Apply dot style to QR code
        // This would require more complex image processing
        // For now, return the original image
        return $image;
    }

    private function applyGradient($image, $options) {
        // Apply gradient effect
        $width = imagesx($image);
        $height = imagesy($image);

        $gradient_image = imagecreatetruecolor($width, $height);

        // Create gradient based on type
        if ($options['gradient_type'] == 'linear') {
            $this->createLinearGradient($gradient_image, $options['gradient_colors']);
        } elseif ($options['gradient_type'] == 'radial') {
            $this->createRadialGradient($gradient_image, $options['gradient_colors']);
        }

        // Merge gradient with QR code
        imagecopymerge($image, $gradient_image, 0, 0, 0, 0, $width, $height, 50);
        imagedestroy($gradient_image);

        return $image;
    }

    private function createLinearGradient($image, $colors) {
        $width = imagesx($image);
        $height = imagesy($image);

        if (count($colors) < 2) return;

        // Convert hex colors to RGB
        $rgb_colors = array_map([$this, 'hexToRgb'], $colors);

        // Create gradient
        for ($i = 0; $i < $width; $i++) {
            $ratio = $i / ($width - 1);
            $color_index = floor($ratio * (count($rgb_colors) - 1));
            $next_index = min($color_index + 1, count($rgb_colors) - 1);
            $color_ratio = fmod($ratio * (count($rgb_colors) - 1), 1);

            $r = $this->interpolate($rgb_colors[$color_index][0], $rgb_colors[$next_index][0], $color_ratio);
            $g = $this->interpolate($rgb_colors[$color_index][1], $rgb_colors[$next_index][1], $color_ratio);
            $b = $this->interpolate($rgb_colors[$color_index][2], $rgb_colors[$next_index][2], $color_ratio);

            $color = imagecolorallocate($image, $r, $g, $b);
            imageline($image, $i, 0, $i, $height - 1, $color);
        }
    }

    private function createRadialGradient($image, $colors) {
        $width = imagesx($image);
        $height = imagesy($image);
        $center_x = $width / 2;
        $center_y = $height / 2;
        $max_distance = sqrt(pow($center_x, 2) + pow($center_y, 2));

        if (count($colors) < 2) return;

        $rgb_colors = array_map([$this, 'hexToRgb'], $colors);

        for ($x = 0; $x < $width; $x++) {
            for ($y = 0; $y < $height; $y++) {
                $distance = sqrt(pow($x - $center_x, 2) + pow($y - $center_y, 2));
                $ratio = $distance / $max_distance;

                $color_index = floor($ratio * (count($rgb_colors) - 1));
                $next_index = min($color_index + 1, count($rgb_colors) - 1);
                $color_ratio = fmod($ratio * (count($rgb_colors) - 1), 1);

                $r = $this->interpolate($rgb_colors[$color_index][0], $rgb_colors[$next_index][0], $color_ratio);
                $g = $this->interpolate($rgb_colors[$color_index][1], $rgb_colors[$next_index][1], $color_ratio);
                $b = $this->interpolate($rgb_colors[$color_index][2], $rgb_colors[$next_index][2], $color_ratio);

                $color = imagecolorallocate($image, $r, $g, $b);
                imagesetpixel($image, $x, $y, $color);
            }
        }
    }

    private function addLogo($image, $logo_path, $qr_size) {
        // Load logo
        $logo_info = getimagesize($logo_path);
        if (!$logo_info) return $image;

        switch ($logo_info[2]) {
            case IMAGETYPE_PNG:
                $logo = imagecreatefrompng($logo_path);
                break;
            case IMAGETYPE_JPEG:
                $logo = imagecreatefromjpeg($logo_path);
                break;
            case IMAGETYPE_GIF:
                $logo = imagecreatefromgif($logo_path);
                break;
            default:
                return $image;
        }

        // Calculate logo size (20% of QR code size)
        $logo_max_size = $qr_size * 0.2;
        $logo_width = imagesx($logo);
        $logo_height = imagesy($logo);

        // Resize logo if needed
        if ($logo_width > $logo_max_size || $logo_height > $logo_max_size) {
            $ratio = min($logo_max_size / $logo_width, $logo_max_size / $logo_height);
            $new_width = $logo_width * $ratio;
            $new_height = $logo_height * $ratio;

            $resized_logo = imagecreatetruecolor($new_width, $new_height);
            imagecopyresampled($resized_logo, $logo, 0, 0, 0, 0, $new_width, $new_height, $logo_width, $logo_height);
            imagedestroy($logo);
            $logo = $resized_logo;
            $logo_width = $new_width;
            $logo_height = $new_height;
        }

        // Calculate position (center of QR code)
        $qr_width = imagesx($image);
        $qr_height = imagesy($image);
        $logo_x = ($qr_width - $logo_width) / 2;
        $logo_y = ($qr_height - $logo_height) / 2;

        // Add white background for logo area
        $white = imagecolorallocate($image, 255, 255, 255);
        imagefilledrectangle($image, $logo_x - 5, $logo_y - 5,
            $logo_x + $logo_width + 5, $logo_y + $logo_height + 5, $white);

        // Merge logo
        imagecopy($image, $logo, $logo_x, $logo_y, 0, 0, $logo_width, $logo_height);

        imagedestroy($logo);
        return $image;
    }

    private function hexToRgb($hex) {
        $hex = str_replace('#', '', $hex);
        if (strlen($hex) == 3) {
            $r = hexdec(str_repeat(substr($hex, 0, 1), 2));
            $g = hexdec(str_repeat(substr($hex, 1, 1), 2));
            $b = hexdec(str_repeat(substr($hex, 2, 1), 2));
        } else {
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
        }
        return [$r, $g, $b];
    }

    private function interpolate($start, $end, $ratio) {
        return $start + ($end - $start) * $ratio;
    }

    // Generate QR code for different types
    public function generateForType($type, $data, $options = []) {
        switch ($type) {
            case 'url':
                $content = $data['url'];
                break;
            case 'text':
                $content = $data['text'];
                break;
            case 'email':
                $content = $this->formatEmailQR($data);
                break;
            case 'sms':
                $content = $this->formatSMSQR($data);
                break;
            case 'wifi':
                $content = $this->formatWifiQR($data);
                break;
            case 'vcard':
                $content = $this->formatVCardQR($data);
                break;
            case 'paypal':
                $content = $this->formatPayPalQR($data);
                break;
            case 'bitcoin':
                $content = $this->formatBitcoinQR($data);
                break;
            case 'social':
                $content = $this->formatSocialQR($data);
                break;
            default:
                $content = $data;
        }

        return $this->generate($content, $options);
    }

    private function formatEmailQR($data) {
        $to = urlencode($data['to'] ?? '');
        $subject = urlencode($data['subject'] ?? '');
        $body = urlencode($data['body'] ?? '');
        return "mailto:{$to}?subject={$subject}&body={$body}";
    }

    private function formatSMSQR($data) {
        $phone = urlencode($data['phone'] ?? '');
        $message = urlencode($data['message'] ?? '');
        return "smsto:{$phone}:{$message}";
    }

    private function formatWifiQR($data) {
        $ssid = $data['ssid'] ?? '';
        $password = $data['password'] ?? '';
        $encryption = $data['encryption'] ?? 'WPA';
        $hidden = isset($data['hidden']) && $data['hidden'] ? 'true' : 'false';
        return "WIFI:T:{$encryption};S:{$ssid};P:{$password};H:{$hidden};;";
    }

    private function formatVCardQR($data) {
        $vcard = "BEGIN:VCARD\nVERSION:3.0\n";
        if (isset($data['name'])) $vcard .= "FN:{$data['name']}\n";
        if (isset($data['company'])) $vcard .= "ORG:{$data['company']}\n";
        if (isset($data['title'])) $vcard .= "TITLE:{$data['title']}\n";
        if (isset($data['phone'])) $vcard .= "TEL:{$data['phone']}\n";
        if (isset($data['email'])) $vcard .= "EMAIL:{$data['email']}\n";
        if (isset($data['website'])) $vcard .= "URL:{$data['website']}\n";
        if (isset($data['address'])) $vcard .= "ADR:{$data['address']}\n";
        if (isset($data['note'])) $vcard .= "NOTE:{$data['note']}\n";
        $vcard .= "END:VCARD";
        return $vcard;
    }

    private function formatPayPalQR($data) {
        $business = urlencode($data['business'] ?? '');
        $amount = $data['amount'] ?? '';
        $currency = $data['currency'] ?? 'USD';
        $item_name = urlencode($data['item_name'] ?? '');
        return "https://www.paypal.com/cgi-bin/webscr?cmd=_xclick&business={$business}&amount={$amount}&currency_code={$currency}&item_name={$item_name}";
    }

    private function formatBitcoinQR($data) {
        $address = $data['address'] ?? '';
        $amount = isset($data['amount']) ? "?amount={$data['amount']}" : '';
        return "bitcoin:{$address}{$amount}";
    }

    private function formatSocialQR($data) {
        $platform = $data['platform'] ?? '';
        $username = $data['username'] ?? '';

        switch (strtolower($platform)) {
            case 'facebook':
                return "https://facebook.com/{$username}";
            case 'twitter':
                return "https://twitter.com/{$username}";
            case 'instagram':
                return "https://instagram.com/{$username}";
            case 'linkedin':
                return "https://linkedin.com/in/{$username}";
            default:
                return "https://{$platform}.com/{$username}";
        }
    }
}
?>