<?php
class ImageProcessor {

    // Resize image
    public static function resize($source_path, $destination_path, $max_width, $max_height, $quality = 85) {
        if (!file_exists($source_path)) {
            return false;
        }

        $image_info = getimagesize($source_path);
        if (!$image_info) {
            return false;
        }

        list($orig_width, $orig_height, $type) = $image_info;

        // Calculate new dimensions
        $ratio = $orig_width / $orig_height;

        if ($max_width / $max_height > $ratio) {
            $new_width = $max_height * $ratio;
            $new_height = $max_height;
        } else {
            $new_width = $max_width;
            $new_height = $max_width / $ratio;
        }

        // Create image from source
        switch ($type) {
            case IMAGETYPE_JPEG:
                $source = imagecreatefromjpeg($source_path);
                break;
            case IMAGETYPE_PNG:
                $source = imagecreatefrompng($source_path);
                break;
            case IMAGETYPE_GIF:
                $source = imagecreatefromgif($source_path);
                break;
            default:
                return false;
        }

        // Create new image
        $new_image = imagecreatetruecolor($new_width, $new_height);

        // Preserve transparency for PNG and GIF
        if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_GIF) {
            imagecolortransparent($new_image, imagecolorallocatealpha($new_image, 0, 0, 0, 127));
            imagealphablending($new_image, false);
            imagesavealpha($new_image, true);
        }

        // Resize image
        imagecopyresampled($new_image, $source, 0, 0, 0, 0,
            $new_width, $new_height, $orig_width, $orig_height);

        // Save image
        switch ($type) {
            case IMAGETYPE_JPEG:
                imagejpeg($new_image, $destination_path, $quality);
                break;
            case IMAGETYPE_PNG:
                imagepng($new_image, $destination_path);
                break;
            case IMAGETYPE_GIF:
                imagegif($new_image, $destination_path);
                break;
        }

        // Clean up
        imagedestroy($source);
        imagedestroy($new_image);

        return true;
    }

    // Crop image to square
    public static function cropToSquare($source_path, $destination_path, $size, $quality = 85) {
        if (!file_exists($source_path)) {
            return false;
        }

        $image_info = getimagesize($source_path);
        if (!$image_info) {
            return false;
        }

        list($orig_width, $orig_height, $type) = $image_info;

        // Calculate crop dimensions
        $min_dimension = min($orig_width, $orig_height);
        $crop_x = ($orig_width - $min_dimension) / 2;
        $crop_y = ($orig_height - $min_dimension) / 2;

        // Create image from source
        switch ($type) {
            case IMAGETYPE_JPEG:
                $source = imagecreatefromjpeg($source_path);
                break;
            case IMAGETYPE_PNG:
                $source = imagecreatefrompng($source_path);
                break;
            case IMAGETYPE_GIF:
                $source = imagecreatefromgif($source_path);
                break;
            default:
                return false;
        }

        // Create new image
        $new_image = imagecreatetruecolor($size, $size);

        // Preserve transparency for PNG and GIF
        if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_GIF) {
            imagecolortransparent($new_image, imagecolorallocatealpha($new_image, 0, 0, 0, 127));
            imagealphablending($new_image, false);
            imagesavealpha($new_image, true);
        }

        // Crop and resize
        imagecopyresampled($new_image, $source, 0, 0, $crop_x, $crop_y,
            $size, $size, $min_dimension, $min_dimension);

        // Save image
        switch ($type) {
            case IMAGETYPE_JPEG:
                imagejpeg($new_image, $destination_path, $quality);
                break;
            case IMAGETYPE_PNG:
                imagepng($new_image, $destination_path);
                break;
            case IMAGETYPE_GIF:
                imagegif($new_image, $destination_path);
                break;
        }

        // Clean up
        imagedestroy($source);
        imagedestroy($new_image);

        return true;
    }

    // Add watermark to image
    public static function addWatermark($source_path, $destination_path, $watermark_path, $position = 'bottom-right', $opacity = 50) {
        if (!file_exists($source_path) || !file_exists($watermark_path)) {
            return false;
        }

        // Load main image
        $source_info = getimagesize($source_path);
        $watermark_info = getimagesize($watermark_path);

        if (!$source_info || !$watermark_info) {
            return false;
        }

        // Create images
        $source = self::createImageFromFile($source_path, $source_info[2]);
        $watermark = self::createImageFromFile($watermark_path, $watermark_info[2]);

        if (!$source || !$watermark) {
            return false;
        }

        // Calculate position
        $source_width = imagesx($source);
        $source_height = imagesy($source);
        $watermark_width = imagesx($watermark);
        $watermark_height = imagesy($watermark);

        switch ($position) {
            case 'top-left':
                $x = 10;
                $y = 10;
                break;
            case 'top-right':
                $x = $source_width - $watermark_width - 10;
                $y = 10;
                break;
            case 'bottom-left':
                $x = 10;
                $y = $source_height - $watermark_height - 10;
                break;
            case 'bottom-right':
                $x = $source_width - $watermark_width - 10;
                $y = $source_height - $watermark_height - 10;
                break;
            case 'center':
                $x = ($source_width - $watermark_width) / 2;
                $y = ($source_height - $watermark_height) / 2;
                break;
            default:
                $x = 10;
                $y = 10;
        }

        // Apply watermark with opacity
        imagecopymerge($source, $watermark, $x, $y, 0, 0,
            $watermark_width, $watermark_height, $opacity);

        // Save image
        self::saveImage($source, $destination_path, $source_info[2]);

        // Clean up
        imagedestroy($source);
        imagedestroy($watermark);

        return true;
    }

    // Convert image to different format
    public static function convertFormat($source_path, $destination_path, $format, $quality = 85) {
        if (!file_exists($source_path)) {
            return false;
        }

        $image_info = getimagesize($source_path);
        if (!$image_info) {
            return false;
        }

        $source = self::createImageFromFile($source_path, $image_info[2]);
        if (!$source) {
            return false;
        }

        // Save in new format
        switch (strtolower($format)) {
            case 'jpg':
            case 'jpeg':
                imagejpeg($source, $destination_path, $quality);
                break;
            case 'png':
                imagepng($source, $destination_path);
                break;
            case 'gif':
                imagegif($source, $destination_path);
                break;
            case 'webp':
                imagewebp($source, $destination_path, $quality);
                break;
            default:
                return false;
        }

        imagedestroy($source);
        return true;
    }

    // Create image from file
    private static function createImageFromFile($path, $type) {
        switch ($type) {
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($path);
            case IMAGETYPE_PNG:
                return imagecreatefrompng($path);
            case IMAGETYPE_GIF:
                return imagecreatefromgif($path);
            case IMAGETYPE_WEBP:
                return imagecreatefromwebp($path);
            default:
                return false;
        }
    }

    // Save image to file
    private static function saveImage($image, $path, $type, $quality = 85) {
        switch ($type) {
            case IMAGETYPE_JPEG:
                return imagejpeg($image, $path, $quality);
            case IMAGETYPE_PNG:
                return imagepng($image, $path);
            case IMAGETYPE_GIF:
                return imagegif($image, $path);
            case IMAGETYPE_WEBP:
                return imagewebp($image, $path, $quality);
            default:
                return false;
        }
    }

    // Validate image file
    public static function validateImage($file_path) {
        if (!file_exists($file_path)) {
            return false;
        }

        $image_info = getimagesize($file_path);
        if (!$image_info) {
            return false;
        }

        $type = $image_info[2];

        // Check if type is allowed
        $allowed_types = [
            IMAGETYPE_JPEG,
            IMAGETYPE_PNG,
            IMAGETYPE_GIF,
            IMAGETYPE_WEBP
        ];

        return in_array($type, $allowed_types);
    }

    // Get image dimensions
    public static function getDimensions($file_path) {
        if (!file_exists($file_path)) {
            return false;
        }

        $image_info = getimagesize($file_path);
        if (!$image_info) {
            return false;
        }

        return [
            'width' => $image_info[0],
            'height' => $image_info[1],
            'type' => $image_info[2],
            'mime' => $image_info['mime']
        ];
    }
}
?>