<?php
require_once __DIR__ . '/../models/User.php';
require_once __DIR__ . '/../middleware/Auth.php';

class UserController {
    private $userModel;
    private $auth;

    public function __construct() {
        $this->userModel = new User();
        $this->auth = new Auth();
    }

    // Register new user
    public function register() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            // Validate input
            $errors = $this->validateRegistration($input);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['errors' => $errors]);
                return;
            }

            // Check if email exists
            $existing = $this->userModel->findByEmail($input['email']);
            if ($existing) {
                http_response_code(400);
                echo json_encode(['error' => 'Email already registered']);
                return;
            }

            // Create user
            $result = $this->userModel->create($input);

            if (!$result) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to create user']);
                return;
            }

            // Start session
            session_start();
            $_SESSION['user_id'] = $result['id'];
            $_SESSION['user_email'] = $input['email'];

            echo json_encode([
                'success' => true,
                'message' => 'Registration successful',
                'data' => [
                    'user_id' => $result['id'],
                    'api_key' => $result['api_key']
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Login user
    public function login() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            if (empty($input['email']) || empty($input['password'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Email and password are required']);
                return;
            }

            // Authenticate user
            $user = $this->userModel->authenticate($input['email'], $input['password']);

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Invalid email or password']);
                return;
            }

            // Start session
            session_start();
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_name'] = $user['full_name'];

            echo json_encode([
                'success' => true,
                'message' => 'Login successful',
                'data' => [
                    'user_id' => $user['id'],
                    'email' => $user['email'],
                    'full_name' => $user['full_name'],
                    'api_key' => $user['api_key']
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Logout user
    public function logout() {
        try {
            session_start();
            session_destroy();

            echo json_encode([
                'success' => true,
                'message' => 'Logout successful'
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Get user profile
    public function profile() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            $userData = $this->userModel->findById($user['id']);

            // Remove sensitive data
            unset($userData['password_hash']);

            echo json_encode([
                'success' => true,
                'data' => $userData
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Update user profile
    public function update() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            // Validate input
            $errors = $this->validateUpdate($input, $user['id']);
            if (!empty($errors)) {
                http_response_code(400);
                echo json_encode(['errors' => $errors]);
                return;
            }

            // Update user
            $success = $this->userModel->update($user['id'], $input);

            if (!$success) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to update profile']);
                return;
            }

            echo json_encode([
                'success' => true,
                'message' => 'Profile updated successfully'
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Change password
    public function changePassword() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            if (empty($input['current_password']) || empty($input['new_password'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Current and new password are required']);
                return;
            }

            // Verify current password
            $dbUser = $this->userModel->findById($user['id']);
            if (!password_verify($input['current_password'], $dbUser['password_hash'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Current password is incorrect']);
                return;
            }

            // Update password
            $success = $this->userModel->update($user['id'], [
                'password' => $input['new_password']
            ]);

            if (!$success) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to change password']);
                return;
            }

            echo json_encode([
                'success' => true,
                'message' => 'Password changed successfully'
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Get user stats
    public function stats() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            $stats = $this->userModel->getStats($user['id']);

            echo json_encode([
                'success' => true,
                'data' => $stats
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Validate registration input
    private function validateRegistration($input) {
        $errors = [];

        if (empty($input['email'])) {
            $errors['email'] = 'Email is required';
        } elseif (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format';
        }

        if (empty($input['password'])) {
            $errors['password'] = 'Password is required';
        } elseif (strlen($input['password']) < 6) {
            $errors['password'] = 'Password must be at least 6 characters';
        }

        if (isset($input['full_name']) && strlen($input['full_name']) > 100) {
            $errors['full_name'] = 'Name must be less than 100 characters';
        }

        return $errors;
    }

    // Validate update input
    private function validateUpdate($input, $user_id) {
        $errors = [];

        if (isset($input['email'])) {
            if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
                $errors['email'] = 'Invalid email format';
            } else {
                // Check if email is taken by another user
                $existing = $this->userModel->findByEmail($input['email']);
                if ($existing && $existing['id'] != $user_id) {
                    $errors['email'] = 'Email already taken';
                }
            }
        }

        if (isset($input['full_name']) && strlen($input['full_name']) > 100) {
            $errors['full_name'] = 'Name must be less than 100 characters';
        }

        return $errors;
    }
}
?>