<?php
require_once __DIR__ . '/../models/QrCode.php';
require_once __DIR__ . '/../models/User.php';
require_once __DIR__ . '/../libraries/QrGenerator.php';
require_once __DIR__ . '/../middleware/Auth.php';

class QrController {
    private $qrModel;
    private $userModel;
    private $qrGenerator;
    private $auth;

    public function __construct() {
        $this->qrModel = new QrCode();
        $this->userModel = new User();
        $this->qrGenerator = new QrGenerator();
        $this->auth = new Auth();
    }

    // Generate QR code
    public function generate() {
        try {
            // Check authentication
            $user = $this->auth->authenticate();

            // Validate input
            $input = json_decode(file_get_contents('php://input'), true);

            if (empty($input['type']) || empty($input['content'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Type and content are required']);
                return;
            }

            // Check if user can create more QR codes
            if ($user && !$this->userModel->canCreateQRCode($user['id'])) {
                http_response_code(403);
                echo json_encode(['error' => 'QR code limit reached. Upgrade your plan.']);
                return;
            }

            // Prepare QR data
            $qrData = [
                'user_id' => $user ? $user['id'] : null,
                'qr_type' => $input['type'],
                'content' => $input['content'],
                'title' => $input['title'] ?? 'Untitled QR Code',
                'settings' => $input['settings'] ?? [],
                'is_dynamic' => $input['is_dynamic'] ?? false,
                'expires_at' => $input['expires_at'] ?? null
            ];

            // Create QR code
            $result = $this->qrModel->create($qrData);

            if (!$result) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to create QR code']);
                return;
            }

            // Save styles if provided
            if (isset($input['style'])) {
                $this->qrModel->saveStyles($result['id'], $input['style']);
            }

            // Generate QR image
            $styles = $input['style'] ?? [];
            $styles['size'] = $styles['size'] ?? QR_DEFAULT_SIZE;
            $styles['margin'] = $styles['margin'] ?? QR_DEFAULT_MARGIN;

            $qrImage = $this->qrGenerator->generate($input['content'], $styles);

            // Save image file
            $filename = $result['public_id'] . '.png';
            $filepath = QRCODE_PATH . $filename;
            imagepng($qrImage, $filepath);
            imagedestroy($qrImage);

            // Return response
            echo json_encode([
                'success' => true,
                'data' => [
                    'id' => $result['id'],
                    'public_id' => $result['public_id'],
                    'image_url' => SITE_URL . 'uploads/qrcodes/' . $filename,
                    'download_url' => SITE_URL . 'api/v1/qr/' . $result['public_id'] . '/download',
                    'edit_url' => SITE_URL . 'qr/' . $result['public_id'] . '/edit'
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Get QR code details
    public function get($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            // Check permissions for private QR codes
            if ($qr['user_id']) {
                $user = $this->auth->authenticate();
                if (!$user || $user['id'] != $qr['user_id']) {
                    http_response_code(403);
                    echo json_encode(['error' => 'Access denied']);
                    return;
                }
            }

            // Get styles
            $styles = $this->qrModel->getStyles($qr['id']);

            echo json_encode([
                'success' => true,
                'data' => [
                    'qr' => $qr,
                    'styles' => $styles
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Update QR code
    public function update($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            // Check authentication and ownership
            $user = $this->auth->authenticate();
            if (!$user || $user['id'] != $qr['user_id']) {
                http_response_code(403);
                echo json_encode(['error' => 'Access denied']);
                return;
            }

            // Only dynamic QR codes can be updated
            if (!$qr['is_dynamic']) {
                http_response_code(400);
                echo json_encode(['error' => 'Only dynamic QR codes can be updated']);
                return;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            // Update QR code
            $updateData = [];
            if (isset($input['content'])) $updateData['content'] = $input['content'];
            if (isset($input['title'])) $updateData['title'] = $input['title'];
            if (isset($input['settings'])) $updateData['settings'] = $input['settings'];

            if (!empty($updateData)) {
                $this->qrModel->update($qr['id'], $updateData);
            }

            // Update styles if provided
            if (isset($input['style'])) {
                $this->qrModel->saveStyles($qr['id'], $input['style']);
            }

            echo json_encode([
                'success' => true,
                'message' => 'QR code updated successfully'
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Delete QR code
    public function delete($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            // Check authentication and ownership
            $user = $this->auth->authenticate();
            if (!$user || $user['id'] != $qr['user_id']) {
                http_response_code(403);
                echo json_encode(['error' => 'Access denied']);
                return;
            }

            // Delete QR code
            $this->qrModel->delete($qr['id']);

            // Delete associated files
            $image_path = QRCODE_PATH . $public_id . '.png';
            if (file_exists($image_path)) {
                unlink($image_path);
            }

            echo json_encode([
                'success' => true,
                'message' => 'QR code deleted successfully'
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // List user's QR codes
    public function list() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            $page = $_GET['page'] ?? 1;
            $limit = $_GET['limit'] ?? 50;
            $offset = ($page - 1) * $limit;

            $qrcodes = $this->qrModel->getUserQRCodes($user['id'], $limit, $offset);

            echo json_encode([
                'success' => true,
                'data' => $qrcodes,
                'pagination' => [
                    'page' => (int)$page,
                    'limit' => (int)$limit,
                    'total' => count($qrcodes)
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Download QR code
    public function download($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            $format = $_GET['format'] ?? 'png';
            $size = $_GET['size'] ?? QR_DEFAULT_SIZE;

            // Get styles
            $styles = $this->qrModel->getStyles($qr['id']);
            $styles['size'] = $size;

            // Generate QR image
            $qrImage = $this->qrGenerator->generate($qr['content'], $styles);

            // Set headers for download
            header('Content-Type: image/png');
            header('Content-Disposition: attachment; filename="' . $qr['title'] . '.png"');

            // Output image
            imagepng($qrImage);
            imagedestroy($qrImage);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Scan QR code (redirect to content)
    public function scan($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            // Check if QR code is expired
            if ($qr['expires_at'] && strtotime($qr['expires_at']) < time()) {
                http_response_code(410);
                echo json_encode(['error' => 'QR code has expired']);
                return;
            }

            // Increment scan count
            $this->qrModel->incrementScanCount($qr['id']);

            // Log analytics
            $analytics = new Analytics();
            $analytics->logScan($qr['id']);

            // Redirect based on QR type
            switch ($qr['qr_type']) {
                case 'url':
                    header('Location: ' . $qr['content']);
                    break;
                case 'text':
                    echo '<pre>' . htmlspecialchars($qr['content']) . '</pre>';
                    break;
                case 'email':
                case 'sms':
                case 'wifi':
                case 'vcard':
                    // These would typically trigger device actions
                    echo json_encode([
                        'type' => $qr['qr_type'],
                        'content' => $qr['content']
                    ]);
                    break;
                default:
                    echo htmlspecialchars($qr['content']);
            }

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }
}
?>