<?php
require_once __DIR__ . '/../models/Analytics.php';
require_once __DIR__ . '/../models/QrCode.php';
require_once __DIR__ . '/../middleware/Auth.php';

class AnalyticsController {
    private $analyticsModel;
    private $qrModel;
    private $auth;

    public function __construct() {
        $this->analyticsModel = new Analytics();
        $this->qrModel = new QrCode();
        $this->auth = new Auth();
    }

    // Get QR code analytics
    public function getQRCodeAnalytics($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            // Check authentication and ownership
            $user = $this->auth->authenticate();
            if (!$user || $user['id'] != $qr['user_id']) {
                http_response_code(403);
                echo json_encode(['error' => 'Access denied']);
                return;
            }

            // Get date range from query parameters
            $start_date = $_GET['start_date'] ?? null;
            $end_date = $_GET['end_date'] ?? null;

            // Validate date format
            if ($start_date && !$this->validateDate($start_date)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid start date format (YYYY-MM-DD)']);
                return;
            }

            if ($end_date && !$this->validateDate($end_date)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid end date format (YYYY-MM-DD)']);
                return;
            }

            // Get analytics
            $analytics = $this->analyticsModel->getQRCodeAnalytics($qr['id'], $start_date, $end_date);

            echo json_encode([
                'success' => true,
                'data' => array_merge($analytics, [
                    'qr_code' => [
                        'title' => $qr['title'],
                        'public_id' => $qr['public_id'],
                        'type' => $qr['qr_type'],
                        'total_scans' => $qr['scan_count'],
                        'created_at' => $qr['created_at']
                    ]
                ])
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Get user analytics
    public function getUserAnalytics() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            // Get date range from query parameters
            $start_date = $_GET['start_date'] ?? null;
            $end_date = $_GET['end_date'] ?? null;

            // Validate date format
            if ($start_date && !$this->validateDate($start_date)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid start date format (YYYY-MM-DD)']);
                return;
            }

            if ($end_date && !$this->validateDate($end_date)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid end date format (YYYY-MM-DD)']);
                return;
            }

            // Get analytics
            $analytics = $this->analyticsModel->getUserAnalytics($user['id'], $start_date, $end_date);

            // Calculate summary
            $total_scans = array_sum(array_column($analytics, 'total_scans'));
            $total_qr_codes = count($analytics);
            $active_qr_codes = count(array_filter($analytics, function($item) {
                return $item['total_scans'] > 0;
            }));

            echo json_encode([
                'success' => true,
                'data' => [
                    'summary' => [
                        'total_qr_codes' => $total_qr_codes,
                        'active_qr_codes' => $active_qr_codes,
                        'total_scans' => $total_scans,
                        'avg_scans_per_code' => $total_qr_codes > 0 ? round($total_scans / $total_qr_codes, 2) : 0
                    ],
                    'qr_codes' => $analytics
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Export analytics as CSV
    public function exportQRCodeAnalytics($public_id) {
        try {
            $qr = $this->qrModel->findByPublicId($public_id);

            if (!$qr) {
                http_response_code(404);
                echo json_encode(['error' => 'QR code not found']);
                return;
            }

            // Check authentication and ownership
            $user = $this->auth->authenticate();
            if (!$user || $user['id'] != $qr['user_id']) {
                http_response_code(403);
                echo json_encode(['error' => 'Access denied']);
                return;
            }

            // Get date range
            $start_date = $_GET['start_date'] ?? null;
            $end_date = $_GET['end_date'] ?? null;

            // Get detailed scan data
            $where = "WHERE qr_code_id = :qr_code_id";
            $params = [':qr_code_id' => $qr['id']];

            if ($start_date) {
                $where .= " AND scan_date >= :start_date";
                $params[':start_date'] = $start_date;
            }

            if ($end_date) {
                $where .= " AND scan_date <= :end_date";
                $params[':end_date'] = $end_date;
            }

            $sql = "SELECT * FROM qr_analytics {$where} ORDER BY scan_date DESC, scan_time DESC";
            $stmt = Database::getInstance()->execute($sql, $params);
            $scans = $stmt->fetchAll();

            // Set CSV headers
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="qr_analytics_' . $public_id . '.csv"');

            // Output CSV
            $output = fopen('php://output', 'w');

            // Header row
            fputcsv($output, [
                'Date', 'Time', 'IP Address', 'Country', 'City',
                'Device Type', 'Browser', 'Operating System', 'Referrer'
            ]);

            // Data rows
            foreach ($scans as $scan) {
                fputcsv($output, [
                    $scan['scan_date'],
                    $scan['scan_time'],
                    $scan['ip_address'],
                    $scan['country'],
                    $scan['city'],
                    $scan['device_type'],
                    $scan['browser'],
                    $scan['os'],
                    $scan['referrer']
                ]);
            }

            fclose($output);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Get real-time analytics (last 24 hours)
    public function getRealTimeAnalytics() {
        try {
            $user = $this->auth->authenticate();

            if (!$user) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                return;
            }

            // Get scans from last 24 hours
            $sql = "SELECT 
                        DATE_FORMAT(a.scan_date, '%Y-%m-%d %H:00:00') as hour,
                        COUNT(*) as scans
                    FROM qr_analytics a
                    INNER JOIN qr_codes q ON a.qr_code_id = q.id
                    WHERE q.user_id = :user_id 
                    AND a.scan_date >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                    GROUP BY hour
                    ORDER BY hour";

            $stmt = Database::getInstance()->execute($sql, [':user_id' => $user['id']]);
            $hourly_scans = $stmt->fetchAll();

            // Get top performing QR codes today
            $sql = "SELECT 
                        q.title,
                        q.public_id,
                        COUNT(a.id) as today_scans
                    FROM qr_codes q
                    LEFT JOIN qr_analytics a ON q.id = a.qr_code_id 
                        AND a.scan_date = CURDATE()
                    WHERE q.user_id = :user_id
                    GROUP BY q.id
                    ORDER BY today_scans DESC
                    LIMIT 5";

            $stmt = Database::getInstance()->execute($sql, [':user_id' => $user['id']]);
            $top_qr_codes = $stmt->fetchAll();

            echo json_encode([
                'success' => true,
                'data' => [
                    'hourly_scans' => $hourly_scans,
                    'top_qr_codes' => $top_qr_codes
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // Validate date format
    private function validateDate($date) {
        $d = DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }
}
?>