<?php
require_once __DIR__ . '/../../app/config/cors.php';
require_once __DIR__ . '/../../app/middleware/Auth.php';
require_once __DIR__ . '/../../app/middleware/RateLimit.php';

// Apply rate limiting
RateLimit::apply('user', 20, 3600); // 20 uploads per hour per user

$auth = new Auth();
$user = $auth->authenticate();

if (!$user) {
    http_response_code(401);
    echo json_encode(['error' => 'Authentication required']);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'POST') {
    try {
        // Check if file was uploaded
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => 'No file uploaded or upload error']);
            exit();
        }

        $file = $_FILES['file'];
        $file_type = $_POST['type'] ?? 'logo'; // 'logo' or 'qrcode'

        // Validate file type
        $allowed_types = ['logo' => ['jpg', 'jpeg', 'png', 'gif', 'svg'],
            'qrcode' => ['png', 'jpg', 'jpeg', 'svg']];

        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        if (!in_array($file_ext, $allowed_types[$file_type] ?? [])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid file type. Allowed: ' . implode(', ', $allowed_types[$file_type])]);
            exit();
        }

        // Validate file size (5MB max for logos, 10MB for QR codes)
        $max_size = $file_type === 'logo' ? 5242880 : 10485760;
        if ($file['size'] > $max_size) {
            http_response_code(400);
            echo json_encode(['error' => 'File too large. Max size: ' . ($max_size / 1024 / 1024) . 'MB']);
            exit();
        }

        // Create unique filename
        $filename = uniqid($user['id'] . '_') . '.' . $file_ext;

        // Determine upload directory
        if ($file_type === 'logo') {
            $upload_dir = LOGO_PATH;
        } else {
            $upload_dir = QRCODE_PATH;
        }

        // Ensure directory exists
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }

        $filepath = $upload_dir . $filename;

        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $filepath)) {
            // If it's an image, optionally resize it
            if (in_array($file_ext, ['jpg', 'jpeg', 'png', 'gif'])) {
                require_once __DIR__ . '/../../app/libraries/ImageProcessor.php';
                $max_dimension = $file_type === 'logo' ? 500 : 2000;

                // Get image dimensions
                list($width, $height) = getimagesize($filepath);

                // Resize if too large
                if ($width > $max_dimension || $height > $max_dimension) {
                    $resized_path = $upload_dir . 'resized_' . $filename;
                    ImageProcessor::resize($filepath, $resized_path, $max_dimension, $max_dimension);

                    // Replace original with resized
                    unlink($filepath);
                    rename($resized_path, $filepath);
                }
            }

            // Return success response
            $relative_path = 'uploads/' . ($file_type === 'logo' ? 'logos/' : 'qrcodes/') . $filename;

            echo json_encode([
                'success' => true,
                'message' => 'File uploaded successfully',
                'data' => [
                    'filename' => $filename,
                    'path' => $relative_path,
                    'full_url' => SITE_URL . $relative_path,
                    'size' => $file['size'],
                    'type' => $file['type']
                ]
            ]);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to save file']);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Internal server error',
            'message' => DEBUG_MODE ? $e->getMessage() : 'An error occurred'
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}
?>