<?php
require_once __DIR__ . '/../../app/config/cors.php';
require_once __DIR__ . '/../../app/middleware/RateLimit.php';
require_once __DIR__ . '/../../app/config/database.php';

// Apply rate limiting
RateLimit::apply('ip', 50, 3600); // 50 requests per hour per IP

$method = $_SERVER['REQUEST_METHOD'];

try {
    $db = Database::getInstance()->getConnection();

    switch ($method) {
        case 'GET':
            // Get all templates
            if (isset($_GET['id'])) {
                // Get specific template
                $sql = "SELECT * FROM qr_templates WHERE id = :id AND is_active = 1";
                $stmt = $db->prepare($sql);
                $stmt->execute([':id' => $_GET['id']]);
                $template = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($template) {
                    $template['settings'] = json_decode($template['settings'], true);
                    echo json_encode([
                        'success' => true,
                        'data' => $template
                    ]);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Template not found']);
                }
            } else {
                // Get all templates with optional filtering
                $category = $_GET['category'] ?? null;
                $premium_only = isset($_GET['premium']) && $_GET['premium'] === 'true';
                $limit = $_GET['limit'] ?? 50;
                $offset = $_GET['offset'] ?? 0;

                $where = "WHERE is_active = 1";
                $params = [];

                if ($category) {
                    $where .= " AND category = :category";
                    $params[':category'] = $category;
                }

                if ($premium_only) {
                    $where .= " AND is_premium = 1";
                }

                // Check if user can see premium templates
                $can_see_premium = false;
                session_start();
                if (isset($_SESSION['user_id'])) {
                    // Check user subscription
                    $user_sql = "SELECT subscription_type FROM users WHERE id = :user_id";
                    $user_stmt = $db->prepare($user_sql);
                    $user_stmt->execute([':user_id' => $_SESSION['user_id']]);
                    $user = $user_stmt->fetch(PDO::FETCH_ASSOC);

                    if ($user && in_array($user['subscription_type'], ['premium', 'enterprise'])) {
                        $can_see_premium = true;
                    }
                }

                if (!$can_see_premium) {
                    $where .= " AND is_premium = 0";
                }

                // Get templates count
                $count_sql = "SELECT COUNT(*) as total FROM qr_templates $where";
                $count_stmt = $db->prepare($count_sql);
                $count_stmt->execute($params);
                $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
                $total = $count_result['total'] ?? 0;

                // Get templates
                $sql = "SELECT * FROM qr_templates $where ORDER BY id DESC LIMIT :limit OFFSET :offset";
                $stmt = $db->prepare($sql);

                $params[':limit'] = (int)$limit;
                $params[':offset'] = (int)$offset;

                foreach ($params as $key => $value) {
                    if ($key === ':limit' || $key === ':offset') {
                        $stmt->bindValue($key, $value, PDO::PARAM_INT);
                    } else {
                        $stmt->bindValue($key, $value);
                    }
                }

                $stmt->execute();
                $templates = $stmt->fetchAll(PDO::FETCH_ASSOC);

                // Decode JSON settings
                foreach ($templates as &$template) {
                    $template['settings'] = json_decode($template['settings'], true);
                }

                echo json_encode([
                    'success' => true,
                    'data' => $templates,
                    'pagination' => [
                        'total' => (int)$total,
                        'limit' => (int)$limit,
                        'offset' => (int)$offset,
                        'has_more' => ($offset + count($templates)) < $total
                    ]
                ]);
            }
            break;

        case 'POST':
            // Create new template (admin only)
            session_start();
            if (!isset($_SESSION['user_id'])) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                break;
            }

            // Check if user is admin
            $user_sql = "SELECT role FROM users WHERE id = :user_id";
            $user_stmt = $db->prepare($user_sql);
            $user_stmt->execute([':user_id' => $_SESSION['user_id']]);
            $user = $user_stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user || $user['role'] !== 'admin') {
                http_response_code(403);
                echo json_encode(['error' => 'Admin access required']);
                break;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            if (empty($input['name']) || empty($input['settings'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Name and settings are required']);
                break;
            }

            $sql = "INSERT INTO qr_templates (name, category, settings, preview_image, is_premium) 
                    VALUES (:name, :category, :settings, :preview_image, :is_premium)";

            $stmt = $db->prepare($sql);
            $result = $stmt->execute([
                ':name' => $input['name'],
                ':category' => $input['category'] ?? null,
                ':settings' => json_encode($input['settings']),
                ':preview_image' => $input['preview_image'] ?? null,
                ':is_premium' => $input['is_premium'] ?? false
            ]);

            if ($result) {
                $template_id = $db->lastInsertId();
                echo json_encode([
                    'success' => true,
                    'message' => 'Template created successfully',
                    'data' => ['id' => $template_id]
                ]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to create template']);
            }
            break;

        case 'PUT':
            // Update template (admin only)
            if (!isset($_GET['id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Template ID required']);
                break;
            }

            session_start();
            if (!isset($_SESSION['user_id'])) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                break;
            }

            // Check if user is admin
            $user_sql = "SELECT role FROM users WHERE id = :user_id";
            $user_stmt = $db->prepare($user_sql);
            $user_stmt->execute([':user_id' => $_SESSION['user_id']]);
            $user = $user_stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user || $user['role'] !== 'admin') {
                http_response_code(403);
                echo json_encode(['error' => 'Admin access required']);
                break;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            $fields = [];
            $params = [':id' => $_GET['id']];

            if (isset($input['name'])) {
                $fields[] = "name = :name";
                $params[':name'] = $input['name'];
            }

            if (isset($input['category'])) {
                $fields[] = "category = :category";
                $params[':category'] = $input['category'];
            }

            if (isset($input['settings'])) {
                $fields[] = "settings = :settings";
                $params[':settings'] = json_encode($input['settings']);
            }

            if (isset($input['preview_image'])) {
                $fields[] = "preview_image = :preview_image";
                $params[':preview_image'] = $input['preview_image'];
            }

            if (isset($input['is_premium'])) {
                $fields[] = "is_premium = :is_premium";
                $params[':is_premium'] = (bool)$input['is_premium'];
            }

            if (isset($input['is_active'])) {
                $fields[] = "is_active = :is_active";
                $params[':is_active'] = (bool)$input['is_active'];
            }

            if (empty($fields)) {
                http_response_code(400);
                echo json_encode(['error' => 'No fields to update']);
                break;
            }

            $sql = "UPDATE qr_templates SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = :id";
            $stmt = $db->prepare($sql);
            $result = $stmt->execute($params);

            if ($result) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Template updated successfully'
                ]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to update template']);
            }
            break;

        case 'DELETE':
            // Delete template (admin only)
            if (!isset($_GET['id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Template ID required']);
                break;
            }

            session_start();
            if (!isset($_SESSION['user_id'])) {
                http_response_code(401);
                echo json_encode(['error' => 'Authentication required']);
                break;
            }

            // Check if user is admin
            $user_sql = "SELECT role FROM users WHERE id = :user_id";
            $user_stmt = $db->prepare($user_sql);
            $user_stmt->execute([':user_id' => $_SESSION['user_id']]);
            $user = $user_stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user || $user['role'] !== 'admin') {
                http_response_code(403);
                echo json_encode(['error' => 'Admin access required']);
                break;
            }

            // Soft delete (set is_active = false)
            $sql = "UPDATE qr_templates SET is_active = 0, updated_at = NOW() WHERE id = :id";
            $stmt = $db->prepare($sql);
            $result = $stmt->execute([':id' => $_GET['id']]);

            if ($result) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Template deleted successfully'
                ]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to delete template']);
            }
            break;

        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Internal server error',
        'message' => DEBUG_MODE ? $e->getMessage() : 'An error occurred'
    ]);

    // Log the error
    error_log("Templates API Error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
}
?>