<?php
require_once __DIR__ . '/../../app/config/cors.php';
require_once __DIR__ . '/../../app/libraries/QrGenerator.php';
require_once __DIR__ . '/../../app/libraries/ImageProcessor.php';
require_once __DIR__ . '/../../app/middleware/RateLimit.php';

// Apply rate limiting
RateLimit::apply('ip', 10, 3600); // 10 downloads per hour per IP

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'POST') {
    try {
        $input = json_decode(file_get_contents('php://input'), true);

        if (empty($input['content'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Content is required']);
            exit();
        }

        $format = $_GET['format'] ?? 'png';
        $qrGenerator = new QrGenerator();

        // Generate QR code
        $qrImage = $qrGenerator->generate($input['content'], $input['style'] ?? []);

        // Set appropriate headers
        $filename = 'qrcode_' . date('Ymd_His') . '.' . $format;

        switch (strtolower($format)) {
            case 'png':
                header('Content-Type: image/png');
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                imagepng($qrImage);
                break;

            case 'jpg':
            case 'jpeg':
                header('Content-Type: image/jpeg');
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                imagejpeg($qrImage, null, 90);
                break;

            case 'svg':
                // For SVG, we need a different approach
                header('Content-Type: image/svg+xml');
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                // SVG generation would be implemented differently
                echo generateSVGQRCode($input['content'], $input['style'] ?? []);
                break;

            case 'pdf':
                header('Content-Type: application/pdf');
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                generatePDFQRCode($input['content'], $input['style'] ?? []);
                break;

            default:
                http_response_code(400);
                echo json_encode(['error' => 'Unsupported format: ' . $format]);
                exit();
        }

        if (is_resource($qrImage)) {
            imagedestroy($qrImage);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Failed to generate QR code',
            'message' => DEBUG_MODE ? $e->getMessage() : 'Please try again'
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}

// Helper function for SVG generation
function generateSVGQRCode($content, $style): string
{
    // Simplified SVG QR code generation
    // In production, use a proper SVG QR library
    $size = $style['size'] ?? 300;
    $fgColor = $style['foreground_color'] ?? '#000000';
    $bgColor = $style['background_color'] ?? '#FFFFFF';

    // This is a placeholder - you'd need a real SVG generator
    $svg = '<?xml version="1.0" encoding="UTF-8"?>
    <!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
    <svg width="' . $size . '" height="' . $size . '" xmlns="http://www.w3.org/2000/svg">
        <rect width="100%" height="100%" fill="' . $bgColor . '"/>
        <text x="50%" y="50%" text-anchor="middle" fill="' . $fgColor . '" font-size="20">
            QR Code: ' . htmlspecialchars(substr($content, 0, 50)) . '...
        </text>
    </svg>';

    return $svg;
}

// Helper function for PDF generation
/**
 * @throws Exception
 */
function generatePDFQRCode($content, $style): void
{
    require_once __DIR__ . '/../../vendor/autoload.php';

    try {
        $size = $style['size'] ?? 300;
        $pdf = new TCPDF();
        $pdf->AddPage();
        $pdf->SetFont('helvetica', '', 12);

        // Generate QR code as image first
        $qrGenerator = new QrGenerator();
        $qrImage = $qrGenerator->generate($content, $style);

        // Save temp image
        $tempFile = tempnam(sys_get_temp_dir(), 'qr_') . '.png';
        imagepng($qrImage, $tempFile);

        // Add to PDF
        $pdf->Image($tempFile, 10, 10, $size/3, $size/3, 'PNG');

        // Add text below
        $pdf->SetY(10 + $size/3 + 10);
        $pdf->Cell(0, 10, 'QR Code Content: ' . substr($content, 0, 100) . '...', 0, 1);
        $pdf->Cell(0, 10, 'Generated: ' . date('Y-m-d H:i:s'), 0, 1);

        // Output PDF
        $pdf->Output('qrcode.pdf', 'D');

        // Cleanup
        unlink($tempFile);
        if (is_resource($qrImage)) {
            imagedestroy($qrImage);
        }

    } catch (Exception $e) {
        throw new Exception('PDF generation failed: ' . $e->getMessage());
    }
}
?>