<?php
/**
 * StyledQR-PHP
 * A server-side alternative to qr-code-styling.js
 */

class StyledQR {
    private $data;
    private $size;
    private $margin;
    private $dotsStyle;
    private $gradient = null;
    private $backgroundColor;

    public function __construct($data, $size = 300) {
        $this->data = $data;
        $this->size = $size;
        $this->margin = 20;
        $this->dotsStyle = 'square';
        $this->backgroundColor = '#ffffff';
    }

    public function setDots($type = 'square') {
        $this->dotsStyle = $type; // options: square, rounded, dots
        return $this;
    }

    public function setGradient($type, $color1, $color2, $rotation = 0) {
        $this->gradient = [
            'type' => $type,
            'color1' => $color1,
            'color2' => $color2,
            'rotation' => $rotation
        ];
        return $this;
    }

    /**
     * Internal: Basic QR Matrix Generator
     * In a production environment, you would use 'chillerlan/php-qrcode'
     * or 'bacon/bacon-qr-code' to generate the $matrix array.
     */
    private function getMatrix() {
        // This is a placeholder for the QR Bit Matrix (1s and 0s)
        // For a full library, integrate a matrix generator here.
        return [
            [1,1,1,0,1],
            [1,0,1,0,0],
            [1,1,1,0,1],
            [0,0,0,0,0],
            [1,0,1,1,1]
        ];
    }

    public function render() {
        $matrix = $this->getMatrix();
        $count = count($matrix);
        $cellSize = ($this->size - ($this->margin * 2)) / $count;

        $svg = '<svg width="'.$this->size.'" height="'.$this->size.'" viewBox="0 0 '.$this->size.' '.$this->size.'" xmlns="http://www.w3.org/2000/svg">';

        // 1. Background
        $svg .= '<rect width="100%" height="100%" fill="'.$this->backgroundColor.'"/>';

        // 2. Define Gradients (if applicable)
        if ($this->gradient) {
            $svg .= '<defs>';
            $svg .= '<linearGradient id="qrGrad" x1="0%" y1="0%" x2="100%" y2="100%">';
            $svg .= '<stop offset="0%" style="stop-color:'.$this->gradient['color1'].';stop-opacity:1" />';
            $svg .= '<stop offset="100%" style="stop-color:'.$this->gradient['color2'].';stop-opacity:1" />';
            $svg .= '</linearGradient></defs>';
        }

        $fill = $this->gradient ? 'url(#qrGrad)' : '#000000';

        // 3. Draw Dots
        for ($y = 0; $y < $count; $y++) {
            for ($x = 0; $x < $count; $x++) {
                if ($matrix[$y][$x] == 1) {
                    $posX = $this->margin + ($x * $cellSize);
                    $posY = $this->margin + ($y * $cellSize);

                    if ($this->dotsStyle === 'dots') {
                        $radius = $cellSize / 2;
                        $svg .= '<circle cx="'.($posX + $radius).'" cy="'.($posY + $radius).'" r="'.$radius.'" fill="'.$fill.'"/>';
                    } elseif ($this->dotsStyle === 'rounded') {
                        $svg .= '<rect x="'.$posX.'" y="'.$posY.'" width="'.$cellSize.'" height="'.$cellSize.'" rx="'.($cellSize/3).'" fill="'.$fill.'"/>';
                    } else {
                        $svg .= '<rect x="'.$posX.'" y="'.$posY.'" width="'.$cellSize.'" height="'.$cellSize.'" fill="'.$fill.'"/>';
                    }
                }
            }
        }

        $svg .= '</svg>';
        return $svg;
    }
}