<?php
// File: QRCodeStyling.php
// QR Code Styling Library for PHP

class QRCodeStyling {
    private $options;
    private $qr;
    private $image;

    // QR Code types
    const MODE_NUMERIC = 1;
    const MODE_ALPHANUMERIC = 2;
    const MODE_BYTE = 4;
    const MODE_KANJI = 8;

    // Error correction levels
    const ERROR_CORRECTION_LOW = 'L';
    const ERROR_CORRECTION_MEDIUM = 'M';
    const ERROR_CORRECTION_QUARTILE = 'Q';
    const ERROR_CORRECTION_HIGH = 'H';

    // Dot types
    const DOTS_SQUARE = 'square';
    const DOTS_DOTS = 'dots';
    const DOTS_ROUNDED = 'rounded';
    const DOTS_CLASSY = 'classy';
    const DOTS_CLASSY_ROUNDED = 'classy-rounded';
    const DOTS_EXTRA_ROUNDED = 'extra-rounded';

    // Corner types
    const CORNERS_SQUARE = 'square';
    const CORNERS_DOT = 'dot';
    const CORNERS_EXTRA_ROUNDED = 'extra-rounded';

    // Gradient types
    const GRADIENT_RADIAL = 'radial';
    const GRADIENT_LINEAR = 'linear';

    // Output types
    const TYPE_PNG = 'png';
    const TYPE_SVG = 'svg';
    const TYPE_JPEG = 'jpeg';
    const TYPE_GIF = 'gif';
    const TYPE_WEBP = 'webp';

    // Default options
    const DEFAULT_OPTIONS = [
        'type' => 'png',
        'width' => 300,
        'height' => 300,
        'data' => '',
        'margin' => 0,
        'qrOptions' => [
            'typeNumber' => 0,
            'mode' => null,
            'errorCorrectionLevel' => 'Q'
        ],
        'imageOptions' => [
            'hideBackgroundDots' => true,
            'imageSize' => 0.4,
            'crossOrigin' => null,
            'margin' => 0
        ],
        'dotsOptions' => [
            'type' => 'square',
            'color' => '#000000'
        ],
        'backgroundOptions' => [
            'color' => '#FFFFFF'
        ],
        'cornersSquareOptions' => null,
        'cornersDotOptions' => null
    ];

    public function __construct($options = null) {
        $this->options = $this->mergeOptions(QRCodeStyling::DEFAULT_OPTIONS, $options ?? []);
        $this->options = $this->validateOptions($this->options);

        // Generate QR code if data is provided
        if (!empty($this->options['data'])) {
            $this->generateQRCode();
        }
    }

    private function mergeOptions($default, $custom) {
        $result = $default;

        foreach ($custom as $key => $value) {
            if (is_array($value) && isset($result[$key]) && is_array($result[$key])) {
                $result[$key] = $this->mergeOptions($result[$key], $value);
            } else {
                $result[$key] = $value;
            }
        }

        return $result;
    }

    private function validateOptions($options) {
        $validated = $options;

        // Validate dimensions
        $validated['width'] = max(1, (int) $validated['width']);
        $validated['height'] = max(1, (int) $validated['height']);
        $validated['margin'] = max(0, (int) $validated['margin']);

        // Ensure margin doesn't exceed dimensions
        $maxMargin = min($validated['width'], $validated['height']);
        if ($validated['margin'] > $maxMargin) {
            $validated['margin'] = $maxMargin;
        }

        // Validate QR options
        if (isset($validated['qrOptions'])) {
            $validated['qrOptions']['typeNumber'] = max(0, min(40, (int) ($validated['qrOptions']['typeNumber'] ?? 0)));

            // Validate error correction level
            $validECLevels = ['L', 'M', 'Q', 'H'];
            if (!in_array($validated['qrOptions']['errorCorrectionLevel'] ?? 'Q', $validECLevels)) {
                $validated['qrOptions']['errorCorrectionLevel'] = 'Q';
            }
        }

        // Validate image options
        if (isset($validated['imageOptions'])) {
            $validated['imageOptions']['hideBackgroundDots'] =
                (bool) ($validated['imageOptions']['hideBackgroundDots'] ?? true);
            $validated['imageOptions']['imageSize'] =
                max(0.1, min(1.0, (float) ($validated['imageOptions']['imageSize'] ?? 0.4)));
            $validated['imageOptions']['margin'] =
                max(0, (int) ($validated['imageOptions']['margin'] ?? 0));
        }

        // Validate gradients
        $gradientFields = ['dotsOptions', 'cornersSquareOptions', 'cornersDotOptions', 'backgroundOptions'];

        foreach ($gradientFields as $field) {
            if (isset($validated[$field]['gradient'])) {
                $gradient = $validated[$field]['gradient'];

                if (!isset($gradient['colorStops']) || empty($gradient['colorStops'])) {
                    throw new InvalidArgumentException("Field 'colorStops' is required in gradient for $field");
                }

                $gradient['rotation'] = isset($gradient['rotation']) ?
                    (float) $gradient['rotation'] : 0;

                foreach ($gradient['colorStops'] as &$stop) {
                    $stop['offset'] = max(0, min(1, (float) $stop['offset']));
                }

                $validated[$field]['gradient'] = $gradient;
            }
        }

        return $validated;
    }

    public function update($options = null) {
        if ($options) {
            $this->options = $this->mergeOptions($this->options, $options);
            $this->options = $this->validateOptions($this->options);
        }

        if (!empty($this->options['data'])) {
            $this->generateQRCode();
        }
    }

    private function generateQRCode() {
        require_once 'QRCodeGenerator.php';

        $qrOptions = $this->options['qrOptions'];

        // Determine mode if not specified
        $mode = $qrOptions['mode'] ?? null;
        if (!$mode) {
            $data = $this->options['data'];
            if (preg_match('/^[0-9]*$/', $data)) {
                $mode = QRCodeStyling::MODE_NUMERIC;
            } elseif (preg_match('/^[0-9A-Z $%*+\-.\/:]*$/', $data)) {
                $mode = QRCodeStyling::MODE_ALPHANUMERIC;
            } else {
                $mode = QRCodeStyling::MODE_BYTE;
            }
        }

        // Generate QR code
        $this->qr = new QRCodeGenerator(
            $qrOptions['typeNumber'],
            $qrOptions['errorCorrectionLevel']
        );

        $this->qr->addData($this->options['data'], $mode);
        $this->qr->make();

        // Load image if specified
        if (!empty($this->options['image'])) {
            $this->loadImage();
        }
    }

    private function loadImage() {
        $imagePath = $this->options['image'];

        if (filter_var($imagePath, FILTER_VALIDATE_URL)) {
            // Download remote image
            $imageData = @file_get_contents($imagePath);
            if ($imageData) {
                $this->image = @imagecreatefromstring($imageData);
            }
        } else {
            // Load local image
            $extension = strtolower(pathinfo($imagePath, PATHINFO_EXTENSION));

            switch ($extension) {
                case 'jpg':
                case 'jpeg':
                    $this->image = @imagecreatefromjpeg($imagePath);
                    break;
                case 'png':
                    $this->image = @imagecreatefrompng($imagePath);
                    break;
                case 'gif':
                    $this->image = @imagecreatefromgif($imagePath);
                    break;
                case 'webp':
                    $this->image = @imagecreatefromwebp($imagePath);
                    break;
                default:
                    $this->image = @imagecreatefromstring(file_get_contents($imagePath));
                    break;
            }
        }

        if (!$this->image) {
            // Create a simple placeholder if image loading fails
            $this->image = imagecreatetruecolor(100, 100);
            $bgColor = imagecolorallocate($this->image, 200, 200, 200);
            imagefill($this->image, 0, 0, $bgColor);
        }
    }

    public function getRawData($format = 'png') {
        if (!$this->qr) {
            throw new RuntimeException("QR code is empty. Call update() with data first.");
        }

        $format = strtolower($format);

        if ($format === 'svg') {
            return $this->generateSVG();
        } else {
            return $this->generateImage($format);
        }
    }

    private function generateSVG() {
        $qrSize = $this->qr->getModuleCount();
        $width = $this->options['width'];
        $height = $this->options['height'];
        $margin = $this->options['margin'];

        // Calculate dot size
        $availableWidth = min($width, $height) - 2 * $margin;
        $dotSize = floor($availableWidth / $qrSize);

        $svg = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $svg .= '<svg width="' . $width . '" height="' . $height . '" ';
        $svg .= 'xmlns="http://www.w3.org/2000/svg" version="1.1">' . PHP_EOL;

        // Draw background
        $svg .= $this->drawSVGBackground($width, $height);

        // Calculate positioning
        $xOffset = floor(($width - $qrSize * $dotSize) / 2);
        $yOffset = floor(($height - $qrSize * $dotSize) / 2);

        // Draw QR code dots
        $svg .= $this->drawSVGDots($xOffset, $yOffset, $dotSize, $qrSize);

        // Draw corners
        $svg .= $this->drawSVGCorners($xOffset, $yOffset, $dotSize, $qrSize);

        // Draw image if specified
        if (!empty($this->options['image']) && $this->image) {
            $svg .= $this->drawSVGImage($xOffset, $yOffset, $dotSize, $qrSize);
        }

        $svg .= '</svg>';

        return $svg;
    }

    private function drawSVGBackground($width, $height) {
        $bgOptions = $this->options['backgroundOptions'];
        $svg = '';

        if (isset($bgOptions['gradient'])) {
            $gradient = $bgOptions['gradient'];
            $gradientId = 'bg-gradient-' . uniqid();

            $svg .= $this->createSVGGradient(
                    $gradient,
                    $gradientId,
                    0, 0, $width, $height
                ) . PHP_EOL;

            $svg .= '<rect x="0" y="0" width="' . $width . '" height="' . $height . '" ';
            $svg .= 'fill="url(#' . $gradientId . ')" />' . PHP_EOL;
        } else {
            $color = $bgOptions['color'] ?? '#FFFFFF';
            $svg .= '<rect x="0" y="0" width="' . $width . '" height="' . $height . '" ';
            $svg .= 'fill="' . $this->escapeSVG($color) . '" />' . PHP_EOL;
        }

        return $svg;
    }

    private function drawSVGDots($xOffset, $yOffset, $dotSize, $qrSize) {
        $dotsOptions = $this->options['dotsOptions'];
        $svg = '';

        // Create clip path for dots
        $clipId = 'dots-clip-' . uniqid();
        $svg .= '<clipPath id="' . $clipId . '">' . PHP_EOL;

        for ($y = 0; $y < $qrSize; $y++) {
            for ($x = 0; $x < $qrSize; $x++) {
                if ($this->qr->isDark($y, $x)) {
                    $svg .= $this->drawSVGDot(
                            $xOffset + $x * $dotSize,
                            $yOffset + $y * $dotSize,
                            $dotSize,
                            $dotsOptions['type']
                        ) . PHP_EOL;
                }
            }
        }

        $svg .= '</clipPath>' . PHP_EOL;

        // Draw colored rectangle with clip
        if (isset($dotsOptions['gradient'])) {
            $gradient = $dotsOptions['gradient'];
            $gradientId = 'dots-gradient-' . uniqid();

            $svg .= $this->createSVGGradient(
                    $gradient,
                    $gradientId,
                    $xOffset, $yOffset,
                    $qrSize * $dotSize,
                    $qrSize * $dotSize
                ) . PHP_EOL;

            $svg .= '<rect x="' . $xOffset . '" y="' . $yOffset . '" ';
            $svg .= 'width="' . ($qrSize * $dotSize) . '" ';
            $svg .= 'height="' . ($qrSize * $dotSize) . '" ';
            $svg .= 'clip-path="url(#' . $clipId . ')" ';
            $svg .= 'fill="url(#' . $gradientId . ')" />' . PHP_EOL;
        } else {
            $color = $dotsOptions['color'] ?? '#000000';
            $svg .= '<rect x="' . $xOffset . '" y="' . $yOffset . '" ';
            $svg .= 'width="' . ($qrSize * $dotSize) . '" ';
            $svg .= 'height="' . ($qrSize * $dotSize) . '" ';
            $svg .= 'clip-path="url(#' . $clipId . ')" ';
            $svg .= 'fill="' . $this->escapeSVG($color) . '" />' . PHP_EOL;
        }

        return $svg;
    }

    private function drawSVGDot($x, $y, $size, $type) {
        switch ($type) {
            case QRCodeStyling::DOTS_DOTS:
                $cx = $x + $size / 2;
                $cy = $y + $size / 2;
                $radius = $size / 2;
                return '<circle cx="' . $cx . '" cy="' . $cy . '" r="' . $radius . '" />';

            case QRCodeStyling::DOTS_ROUNDED:
                $radius = $size / 2;
                return '<rect x="' . $x . '" y="' . $y . '" width="' . $size . '" height="' . $size . '" rx="' . $radius . '" ry="' . $radius . '" />';

            case QRCodeStyling::DOTS_EXTRA_ROUNDED:
                $radius = $size * 0.4;
                return '<rect x="' . $x . '" y="' . $y . '" width="' . $size . '" height="' . $size . '" rx="' . $radius . '" ry="' . $radius . '" />';

            case QRCodeStyling::DOTS_CLASSY:
                // Simplified classy shape
                $path = 'M ' . $x . ' ' . ($y + $size/2);
                $path .= ' L ' . ($x + $size/2) . ' ' . $y;
                $path .= ' L ' . ($x + $size) . ' ' . ($y + $size/2);
                $path .= ' L ' . ($x + $size/2) . ' ' . ($y + $size);
                $path .= ' Z';
                return '<path d="' . $path . '" />';

            case QRCodeStyling::DOTS_CLASSY_ROUNDED:
                $radius = $size / 4;
                return '<rect x="' . $x . '" y="' . $y . '" width="' . $size . '" height="' . $size . '" rx="' . $radius . '" ry="' . $radius . '" />';

            case QRCodeStyling::DOTS_SQUARE:
            default:
                return '<rect x="' . $x . '" y="' . $y . '" width="' . $size . '" height="' . $size . '" />';
        }
    }

    private function drawSVGCorners($xOffset, $yOffset, $dotSize, $qrSize) {
        $svg = '';
        $cornerSize = 7; // Standard QR code corner size

        // Positions: top-left, top-right, bottom-left
        $positions = [
            [0, 0, 0],
            [1, 0, 90],
            [0, 1, -90]
        ];

        foreach ($positions as $pos) {
            list($xMultiplier, $yMultiplier, $rotation) = $pos;

            $x = $xOffset + $xMultiplier * $dotSize * ($qrSize - $cornerSize);
            $y = $yOffset + $yMultiplier * $dotSize * ($qrSize - $cornerSize);

            $svg .= $this->drawSVGCornerSquare($x, $y, $dotSize, $cornerSize, $rotation);
            $svg .= $this->drawSVGCornerDot($x + 2 * $dotSize, $y + 2 * $dotSize, $dotSize, 3, $rotation);
        }

        return $svg;
    }

    private function drawSVGCornerSquare($x, $y, $dotSize, $size, $rotation) {
        $svg = '';
        $cornerOptions = $this->options['cornersSquareOptions'] ?? $this->options['dotsOptions'];
        $type = $cornerOptions['type'] ?? QRCodeStyling::DOTS_SQUARE;

        // Create group with rotation
        $groupId = 'corner-square-' . uniqid();
        $svg .= '<g id="' . $groupId . '"';
        if ($rotation != 0) {
            $centerX = $x + ($size * $dotSize) / 2;
            $centerY = $y + ($size * $dotSize) / 2;
            $svg .= ' transform="rotate(' . $rotation . ', ' . $centerX . ', ' . $centerY . ')"';
        }
        $svg .= '>' . PHP_EOL;

        // Draw corner square pattern
        for ($i = 0; $i < $size; $i++) {
            for ($j = 0; $j < $size; $j++) {
                // Standard corner pattern (filled border, empty center)
                if (($i == 0 || $i == $size-1 || $j == 0 || $j == $size-1) ||
                    ($i >= 2 && $i <= 4 && $j >= 2 && $j <= 4)) {
                    $svg .= $this->drawSVGDot(
                            $x + $i * $dotSize,
                            $y + $j * $dotSize,
                            $dotSize,
                            $type
                        ) . PHP_EOL;
                }
            }
        }

        $svg .= '</g>' . PHP_EOL;

        return $svg;
    }

    private function drawSVGCornerDot($x, $y, $dotSize, $size, $rotation) {
        $svg = '';
        $cornerOptions = $this->options['cornersDotOptions'] ?? $this->options['dotsOptions'];
        $type = $cornerOptions['type'] ?? QRCodeStyling::DOTS_SQUARE;

        // Create group with rotation
        $groupId = 'corner-dot-' . uniqid();
        $svg .= '<g id="' . $groupId . '"';
        if ($rotation != 0) {
            $centerX = $x + ($size * $dotSize) / 2;
            $centerY = $y + ($size * $dotSize) / 2;
            $svg .= ' transform="rotate(' . $rotation . ', ' . $centerX . ', ' . $centerY . ')"';
        }
        $svg .= '>' . PHP_EOL;

        // Draw corner dot pattern (3x3 filled square)
        for ($i = 0; $i < $size; $i++) {
            for ($j = 0; $j < $size; $j++) {
                $svg .= $this->drawSVGDot(
                        $x + $i * $dotSize,
                        $y + $j * $dotSize,
                        $dotSize,
                        $type
                    ) . PHP_EOL;
            }
        }

        $svg .= '</g>' . PHP_EOL;

        return $svg;
    }

    private function drawSVGImage($xOffset, $yOffset, $dotSize, $qrSize) {
        if (!$this->image) {
            return '';
        }

        $imageOptions = $this->options['imageOptions'];
        $imageWidth = imagesx($this->image);
        $imageHeight = imagesy($this->image);

        // Calculate optimal image dimensions
        $errorCorrection = $this->options['qrOptions']['errorCorrectionLevel'];
        $maxHiddenDots = $this->calculateMaxHiddenDots($qrSize, $errorCorrection);

        $imageSize = $this->calculateImageSize(
            $imageWidth,
            $imageHeight,
            $maxHiddenDots,
            $qrSize - 14,
            $dotSize
        );

        $centerX = $xOffset + ($qrSize * $dotSize) / 2;
        $centerY = $yOffset + ($qrSize * $dotSize) / 2;
        $imageX = $centerX - $imageSize['width'] / 2;
        $imageY = $centerY - $imageSize['height'] / 2;

        // Apply margin
        $imageX += $imageOptions['margin'];
        $imageY += $imageOptions['margin'];
        $finalWidth = max(1, $imageSize['width'] - 2 * $imageOptions['margin']);
        $finalHeight = max(1, $imageSize['height'] - 2 * $imageOptions['margin']);

        // Convert image to data URL
        $imageData = $this->imageToDataURL($this->options['image']);
        if (!$imageData) {
            return '';
        }

        $svg = '<image href="' . $this->escapeSVG($imageData) . '" ';
        $svg .= 'x="' . $imageX . '" y="' . $imageY . '" ';
        $svg .= 'width="' . $finalWidth . '" height="' . $finalHeight . '" ';
        $svg .= 'preserveAspectRatio="xMidYMid meet" />' . PHP_EOL;

        return $svg;
    }

    private function createSVGGradient($gradient, $id, $x, $y, $width, $height) {
        $type = $gradient['type'] ?? QRCodeStyling::GRADIENT_LINEAR;
        $rotation = $gradient['rotation'] ?? 0;

        $svg = '';

        if ($type === QRCodeStyling::GRADIENT_RADIAL) {
            $cx = $x + $width / 2;
            $cy = $y + $height / 2;
            $radius = max($width, $height) / 2;

            $svg .= '<radialGradient id="' . $id . '" gradientUnits="userSpaceOnUse" ';
            $svg .= 'cx="' . $cx . '" cy="' . $cy . '" r="' . $radius . '">' . PHP_EOL;
        } else {
            // Linear gradient with rotation
            $angle = deg2rad($rotation);

            $x1 = $x + $width / 2;
            $y1 = $y + $height / 2;
            $x2 = $x + $width / 2;
            $y2 = $y + $height / 2;

            // Calculate gradient vector based on angle
            $dx = cos($angle) * ($width / 2);
            $dy = sin($angle) * ($height / 2);

            $x1 -= $dx;
            $y1 -= $dy;
            $x2 += $dx;
            $y2 += $dy;

            $svg .= '<linearGradient id="' . $id . '" gradientUnits="userSpaceOnUse" ';
            $svg .= 'x1="' . round($x1) . '" y1="' . round($y1) . '" ';
            $svg .= 'x2="' . round($x2) . '" y2="' . round($y2) . '">' . PHP_EOL;
        }

        // Add color stops
        foreach ($gradient['colorStops'] as $stop) {
            $offset = $stop['offset'] * 100;
            $color = $stop['color'];
            $svg .= '<stop offset="' . $offset . '%" ';
            $svg .= 'stop-color="' . $this->escapeSVG($color) . '" />' . PHP_EOL;
        }

        $svg .= '</' . ($type === QRCodeStyling::GRADIENT_RADIAL ? 'radialGradient' : 'linearGradient') . '>' . PHP_EOL;

        return $svg;
    }

    private function generateImage($format = 'png') {
        if (!extension_loaded('gd')) {
            throw new RuntimeException('GD library is not installed or enabled.');
        }

        $qrSize = $this->qr->getModuleCount();
        $width = $this->options['width'];
        $height = $this->options['height'];
        $margin = $this->options['margin'];

        // Create image
        $image = imagecreatetruecolor($width, $height);
        if (!$image) {
            throw new RuntimeException('Failed to create image.');
        }

        // Enable alpha blending for PNG
        if ($format === 'png') {
            imagesavealpha($image, true);
            $transparent = imagecolorallocatealpha($image, 0, 0, 0, 127);
            imagefill($image, 0, 0, $transparent);
        }

        // Draw background
        $this->drawImageBackground($image, $width, $height);

        // Calculate dot size
        $availableWidth = min($width, $height) - 2 * $margin;
        $dotSize = floor($availableWidth / $qrSize);
        if ($dotSize < 1) $dotSize = 1;

        // Calculate positioning
        $xOffset = floor(($width - $qrSize * $dotSize) / 2);
        $yOffset = floor(($height - $qrSize * $dotSize) / 2);

        // Draw QR code
        $this->drawImageDots($image, $xOffset, $yOffset, $dotSize, $qrSize);

        // Draw corners
        $this->drawImageCorners($image, $xOffset, $yOffset, $dotSize, $qrSize);

        // Draw image if specified
        if (!empty($this->options['image']) && $this->image) {
            $this->drawImageOverlay($image, $xOffset, $yOffset, $dotSize, $qrSize);
        }

        // Output image
        ob_start();

        switch ($format) {
            case 'jpg':
            case 'jpeg':
                imagejpeg($image, null, 90);
                break;
            case 'gif':
                imagegif($image);
                break;
            case 'webp':
                imagewebp($image, null, 90);
                break;
            case 'png':
            default:
                imagepng($image, null, 9);
                break;
        }

        $imageData = ob_get_clean();
        imagedestroy($image);

        return $imageData;
    }

    private function drawImageBackground($image, $width, $height) {
        $bgOptions = $this->options['backgroundOptions'];

        if (isset($bgOptions['gradient'])) {
            $this->drawImageGradient($image, $bgOptions['gradient'], 0, 0, $width, $height);
        } else {
            $color = $this->hexToColor($image, $bgOptions['color'] ?? '#FFFFFF');
            imagefilledrectangle($image, 0, 0, $width - 1, $height - 1, $color);
        }
    }

    private function drawImageGradient($image, $gradient, $x, $y, $width, $height) {
        $type = $gradient['type'] ?? QRCodeStyling::GRADIENT_LINEAR;
        $rotation = $gradient['rotation'] ?? 0;
        $stops = $gradient['colorStops'];

        if (count($stops) === 1) {
            $color = $this->hexToColor($image, $stops[0]['color']);
            imagefilledrectangle($image, $x, $y, $x + $width - 1, $y + $height - 1, $color);
            return;
        }

        // Sort stops by offset
        usort($stops, function($a, $b) {
            return $a['offset'] <=> $b['offset'];
        });

        if ($type === QRCodeStyling::GRADIENT_LINEAR) {
            $this->drawLinearGradient($image, $stops, $x, $y, $width, $height, $rotation);
        } else {
            $this->drawRadialGradient($image, $stops, $x, $y, $width, $height);
        }
    }

    private function drawLinearGradient($image, $stops, $x, $y, $width, $height, $rotation) {
        $angle = deg2rad($rotation);

        // Calculate gradient direction
        $dx = cos($angle);
        $dy = sin($angle);

        // Create gradient image
        $gradientImage = imagecreatetruecolor($width, $height);
        imagesavealpha($gradientImage, true);
        $transparent = imagecolorallocatealpha($gradientImage, 0, 0, 0, 127);
        imagefill($gradientImage, 0, 0, $transparent);

        // Draw gradient lines
        for ($i = 0; $i < count($stops) - 1; $i++) {
            $start = $stops[$i];
            $end = $stops[$i + 1];

            $startColor = $this->hexToColor($gradientImage, $start['color']);
            $endColor = $this->hexToColor($gradientImage, $end['color']);

            $startPos = $start['offset'];
            $endPos = $end['offset'];

            // Calculate positions along gradient line
            $steps = 100;
            for ($j = 0; $j <= $steps; $j++) {
                $ratio = $j / $steps;
                $pos = $startPos + ($endPos - $startPos) * $ratio;

                // Calculate color at this position
                $color = $this->interpolateColor($gradientImage, $startColor, $endColor, $ratio);

                // Calculate line position
                $lineX = $x + $dx * $pos * $width;
                $lineY = $y + $dy * $pos * $height;

                // Draw gradient line (simplified - actual implementation would be more complex)
                imageline($gradientImage,
                    max(0, min($width-1, $lineX)),
                    max(0, min($height-1, $lineY)),
                    max(0, min($width-1, $lineX + $dx * 10)),
                    max(0, min($height-1, $lineY + $dy * 10)),
                    $color
                );
            }
        }

        // Apply gradient to main image
        imagecopy($image, $gradientImage, $x, $y, 0, 0, $width, $height);
        imagedestroy($gradientImage);
    }

    private function drawRadialGradient($image, $stops, $x, $y, $width, $height) {
        $centerX = $x + $width / 2;
        $centerY = $y + $height / 2;
        $maxRadius = max($width, $height) / 2;

        // Create gradient image
        $gradientImage = imagecreatetruecolor($width, $height);
        imagesavealpha($gradientImage, true);
        $transparent = imagecolorallocatealpha($gradientImage, 0, 0, 0, 127);
        imagefill($gradientImage, 0, 0, $transparent);

        // Draw concentric circles
        for ($i = 0; $i < count($stops) - 1; $i++) {
            $start = $stops[$i];
            $end = $stops[$i + 1];

            $startColor = $this->hexToColor($gradientImage, $start['color']);
            $endColor = $this->hexToColor($gradientImage, $end['color']);

            $startRadius = $start['offset'] * $maxRadius;
            $endRadius = $end['offset'] * $maxRadius;

            $radiusSteps = max(1, $endRadius - $startRadius);
            for ($r = 0; $r <= $radiusSteps; $r++) {
                $ratio = $r / $radiusSteps;
                $currentRadius = $startRadius + $r;
                $color = $this->interpolateColor($gradientImage, $startColor, $endColor, $ratio);

                if ($currentRadius > 0) {
                    imagefilledellipse($gradientImage,
                        $centerX - $x,
                        $centerY - $y,
                        $currentRadius * 2,
                        $currentRadius * 2,
                        $color
                    );
                }
            }
        }

        // Apply gradient to main image
        imagecopy($image, $gradientImage, $x, $y, 0, 0, $width, $height);
        imagedestroy($gradientImage);
    }

    private function drawImageDots($image, $xOffset, $yOffset, $dotSize, $qrSize) {
        $dotsOptions = $this->options['dotsOptions'];

        // Create dots color
        if (isset($dotsOptions['gradient'])) {
            // For gradients, use first color as fallback
            $firstColor = $dotsOptions['gradient']['colorStops'][0]['color'] ?? '#000000';
            $dotsColor = $this->hexToColor($image, $firstColor);
        } else {
            $dotsColor = $this->hexToColor($image, $dotsOptions['color'] ?? '#000000');
        }

        // Draw each dot
        for ($y = 0; $y < $qrSize; $y++) {
            for ($x = 0; $x < $qrSize; $x++) {
                if ($this->qr->isDark($y, $x)) {
                    $this->drawImageDot(
                        $image,
                        $xOffset + $x * $dotSize,
                        $yOffset + $y * $dotSize,
                        $dotSize,
                        $dotsOptions['type'],
                        $dotsColor
                    );
                }
            }
        }
    }

    private function drawImageDot($image, $x, $y, $size, $type, $color) {
        switch ($type) {
            case QRCodeStyling::DOTS_DOTS:
                $centerX = $x + $size / 2;
                $centerY = $y + $size / 2;
                $radius = $size / 2;
                imagefilledellipse($image, $centerX, $centerY, $size, $size, $color);
                break;

            case QRCodeStyling::DOTS_ROUNDED:
                $this->drawRoundedRectangle($image, $x, $y, $size, $size, $size/2, $color);
                break;

            case QRCodeStyling::DOTS_EXTRA_ROUNDED:
                $radius = $size * 0.4;
                $this->drawRoundedRectangle($image, $x, $y, $size, $size, $radius, $color);
                break;

            case QRCodeStyling::DOTS_CLASSY:
                // Draw diamond shape for classy
                $points = [
                    $x + $size/2, $y,           // Top
                    $x + $size, $y + $size/2,    // Right
                    $x + $size/2, $y + $size,    // Bottom
                    $x, $y + $size/2            // Left
                ];
                imagefilledpolygon($image, $points, 4, $color);
                break;

            case QRCodeStyling::DOTS_CLASSY_ROUNDED:
                $radius = $size / 4;
                $this->drawRoundedRectangle($image, $x, $y, $size, $size, $radius, $color);
                break;

            case QRCodeStyling::DOTS_SQUARE:
            default:
                imagefilledrectangle($image, $x, $y, $x + $size - 1, $y + $size - 1, $color);
                break;
        }
    }

    private function drawRoundedRectangle($image, $x, $y, $width, $height, $radius, $color) {
        // Draw rounded rectangle using circles and rectangles
        $diameter = $radius * 2;

        // Draw corners
        imagefilledellipse($image, $x + $radius, $y + $radius, $diameter, $diameter, $color);
        imagefilledellipse($image, $x + $width - $radius - 1, $y + $radius, $diameter, $diameter, $color);
        imagefilledellipse($image, $x + $radius, $y + $height - $radius - 1, $diameter, $diameter, $color);
        imagefilledellipse($image, $x + $width - $radius - 1, $y + $height - $radius - 1, $diameter, $diameter, $color);

        // Draw rectangles
        imagefilledrectangle($image, $x + $radius, $y, $x + $width - $radius - 1, $y + $height - 1, $color);
        imagefilledrectangle($image, $x, $y + $radius, $x + $width - 1, $y + $height - $radius - 1, $color);
    }

    private function drawImageCorners($image, $xOffset, $yOffset, $dotSize, $qrSize) {
        $cornerSize = 7;

        // Positions: top-left, top-right, bottom-left
        $positions = [
            [0, 0],
            [1, 0],
            [0, 1]
        ];

        foreach ($positions as $pos) {
            list($xMultiplier, $yMultiplier) = $pos;

            $x = $xOffset + $xMultiplier * $dotSize * ($qrSize - $cornerSize);
            $y = $yOffset + $yMultiplier * $dotSize * ($qrSize - $cornerSize);

            $this->drawCornerSquare($image, $x, $y, $dotSize, $cornerSize);
            $this->drawCornerDot($image, $x + 2 * $dotSize, $y + 2 * $dotSize, $dotSize, 3);
        }
    }

    private function drawCornerSquare($image, $x, $y, $dotSize, $size) {
        $cornerOptions = $this->options['cornersSquareOptions'] ?? $this->options['dotsOptions'];
        $type = $cornerOptions['type'] ?? QRCodeStyling::DOTS_SQUARE;

        // Use dots color for corners if no specific color is set
        $dotsColor = $this->hexToColor($image, $this->options['dotsOptions']['color'] ?? '#000000');

        for ($i = 0; $i < $size; $i++) {
            for ($j = 0; $j < $size; $j++) {
                if (($i == 0 || $i == $size-1 || $j == 0 || $j == $size-1) ||
                    ($i >= 2 && $i <= 4 && $j >= 2 && $j <= 4)) {
                    $this->drawImageDot(
                        $image,
                        $x + $i * $dotSize,
                        $y + $j * $dotSize,
                        $dotSize,
                        $type,
                        $dotsColor
                    );
                }
            }
        }
    }

    private function drawCornerDot($image, $x, $y, $dotSize, $size) {
        $cornerOptions = $this->options['cornersDotOptions'] ?? $this->options['dotsOptions'];
        $type = $cornerOptions['type'] ?? QRCodeStyling::DOTS_SQUARE;

        // Use dots color for corners if no specific color is set
        $dotsColor = $this->hexToColor($image, $this->options['dotsOptions']['color'] ?? '#000000');

        for ($i = 0; $i < $size; $i++) {
            for ($j = 0; $j < $size; $j++) {
                $this->drawImageDot(
                    $image,
                    $x + $i * $dotSize,
                    $y + $j * $dotSize,
                    $dotSize,
                    $type,
                    $dotsColor
                );
            }
        }
    }

    private function drawImageOverlay($image, $xOffset, $yOffset, $dotSize, $qrSize) {
        if (!$this->image) {
            return;
        }

        $imageOptions = $this->options['imageOptions'];

        // Calculate image size
        $errorCorrection = $this->options['qrOptions']['errorCorrectionLevel'];
        $maxHiddenDots = $this->calculateMaxHiddenDots($qrSize, $errorCorrection);

        $origWidth = imagesx($this->image);
        $origHeight = imagesy($this->image);

        $imageSize = $this->calculateImageSize(
            $origWidth,
            $origHeight,
            $maxHiddenDots,
            $qrSize - 14,
            $dotSize
        );

        // Calculate position
        $centerX = $xOffset + ($qrSize * $dotSize) / 2;
        $centerY = $yOffset + ($qrSize * $dotSize) / 2;
        $imageX = $centerX - $imageSize['width'] / 2 + $imageOptions['margin'];
        $imageY = $centerY - $imageSize['height'] / 2 + $imageOptions['margin'];

        $overlayWidth = max(1, $imageSize['width'] - 2 * $imageOptions['margin']);
        $overlayHeight = max(1, $imageSize['height'] - 2 * $imageOptions['margin']);

        // Resize and overlay image
        $resizedOverlay = imagecreatetruecolor($overlayWidth, $overlayHeight);
        if ($overlayWidth > 0 && $overlayHeight > 0) {
            imagecopyresampled(
                $resizedOverlay,
                $this->image,
                0, 0, 0, 0,
                $overlayWidth,
                $overlayHeight,
                $origWidth,
                $origHeight
            );

            imagecopy(
                $image,
                $resizedOverlay,
                $imageX,
                $imageY,
                0,
                0,
                $overlayWidth,
                $overlayHeight
            );
        }

        imagedestroy($resizedOverlay);
    }

    private function calculateMaxHiddenDots($qrSize, $errorCorrection) {
        $ecLevels = [
            'L' => 0.07,
            'M' => 0.15,
            'Q' => 0.25,
            'H' => 0.3
        ];

        $ecFactor = $ecLevels[$errorCorrection] ?? 0.25;
        $imageSize = $this->options['imageOptions']['imageSize'] ?? 0.4;

        return floor($imageSize * $ecFactor * $qrSize * $qrSize);
    }

    private function calculateImageSize($origWidth, $origHeight, $maxHiddenDots, $maxHiddenAxisDots, $dotSize) {
        if ($origWidth <= 0 || $origHeight <= 0 || $maxHiddenDots <= 0 || $dotSize <= 0) {
            return ['width' => 0, 'height' => 0, 'hideXDots' => 0, 'hideYDots' => 0];
        }

        $aspectRatio = $origHeight / $origWidth;

        // Calculate initial dimensions
        $hideXDots = floor(sqrt($maxHiddenDots / $aspectRatio));
        if ($hideXDots <= 0) $hideXDots = 1;
        if ($maxHiddenAxisDots && $maxHiddenAxisDots < $hideXDots) {
            $hideXDots = $maxHiddenAxisDots;
        }

        // Make odd number
        if ($hideXDots % 2 == 0) $hideXDots--;

        $width = $hideXDots * $dotSize;
        $hideYDots = 1 + 2 * ceil(($hideXDots * $aspectRatio - 1) / 2);
        $height = round($width * $aspectRatio);

        // Adjust if too large
        if ($hideYDots * $hideXDots > $maxHiddenDots || ($maxHiddenAxisDots && $maxHiddenAxisDots < $hideYDots)) {
            if ($maxHiddenAxisDots && $maxHiddenAxisDots < $hideYDots) {
                $hideYDots = $maxHiddenAxisDots;
                if ($hideYDots % 2 == 0) $hideYDots--;
            } else {
                $hideYDots -= 2;
            }

            $height = $hideYDots * $dotSize;
            $hideXDots = 1 + 2 * ceil(($hideYDots / $aspectRatio - 1) / 2);
            $width = round($height / $aspectRatio);
        }

        return [
            'width' => $width,
            'height' => $height,
            'hideXDots' => $hideXDots,
            'hideYDots' => $hideYDots
        ];
    }

    private function hexToColor($image, $hex) {
        $hex = str_replace('#', '', $hex);

        if (strlen($hex) == 3) {
            $r = hexdec(str_repeat(substr($hex, 0, 1), 2));
            $g = hexdec(str_repeat(substr($hex, 1, 1), 2));
            $b = hexdec(str_repeat(substr($hex, 2, 1), 2));
        } else {
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
        }

        return imagecolorallocate($image, $r, $g, $b);
    }

    private function interpolateColor($image, $color1, $color2, $ratio) {
        $r1 = ($color1 >> 16) & 0xFF;
        $g1 = ($color1 >> 8) & 0xFF;
        $b1 = $color1 & 0xFF;

        $r2 = ($color2 >> 16) & 0xFF;
        $g2 = ($color2 >> 8) & 0xFF;
        $b2 = $color2 & 0xFF;

        $r = $r1 + ($r2 - $r1) * $ratio;
        $g = $g1 + ($g2 - $g1) * $ratio;
        $b = $b1 + ($b2 - $b1) * $ratio;

        return imagecolorallocate($image, $r, $g, $b);
    }

    private function imageToDataURL($path) {
        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }

        if (file_exists($path)) {
            $imageData = file_get_contents($path);
            if ($imageData) {
                $mime = mime_content_type($path);
                $base64 = base64_encode($imageData);
                return 'data:' . $mime . ';base64,' . $base64;
            }
        }

        return '';
    }

    private function escapeSVG($string): string
    {
        return htmlspecialchars($string, ENT_QUOTES | ENT_XML1, 'UTF-8');
    }

    public function download($options = []) {
        if (!$this->qr) {
            throw new RuntimeException("QR code is empty. Call update() with data first.");
        }

        $filename = $options['name'] ?? 'qr_code';
        $extension = $options['extension'] ?? 'png';

        $format = strtolower($extension);
        $data = $this->getRawData($format);

        // Set headers for download
        header('Content-Type: ' . $this->getMimeType($format));
        header('Content-Disposition: attachment; filename="' . $filename . '.' . $extension . '"');
        header('Content-Length: ' . strlen($data));

        echo $data;
        exit;
    }

    public function display($format = 'png') {
        if (!$this->qr) {
            throw new RuntimeException("QR code is empty. Call update() with data first.");
        }

        $format = strtolower($format);
        $data = $this->getRawData($format);

        header('Content-Type: ' . $this->getMimeType($format));
        header('Cache-Control: max-age=3600, public');

        echo $data;
        exit;
    }

    private function getMimeType($format): string
    {
        switch ($format) {
            case 'jpg':
            case 'jpeg':
                return 'image/jpeg';
            case 'gif':
                return 'image/gif';
            case 'webp':
                return 'image/webp';
            case 'svg':
                return 'image/svg+xml';
            case 'png':
            default:
                return 'image/png';
        }
    }

    public static function clearContainer($container) {
        if ($container && is_string($container) && file_exists($container)) {
            file_put_contents($container, '');
        }
    }

    public function getOptions() {
        return $this->options;
    }

    public function getQRCode() {
        return $this->qr;
    }
}