<?php
// File: QRCodeGenerator.php
// QR Code Generator Core

class QRCodeGenerator {
    private $typeNumber;
    private $errorCorrectLevel;
    private $modules;
    private $moduleCount;
    private $dataCache;
    private $dataList;

    const PATTERN_POSITION_TABLE = [
        [],
        [6, 18],
        [6, 22],
        [6, 26],
        [6, 30],
        [6, 34],
        [6, 22, 38],
        [6, 24, 42],
        [6, 26, 46],
        [6, 28, 50],
        [6, 30, 54],
        [6, 32, 58],
        [6, 34, 62],
        [6, 26, 46, 66],
        [6, 26, 48, 70],
        [6, 26, 50, 74],
        [6, 30, 54, 78],
        [6, 30, 56, 82],
        [6, 30, 58, 86],
        [6, 34, 62, 90],
        [6, 28, 50, 72, 94],
        [6, 26, 50, 74, 98],
        [6, 30, 54, 78, 102],
        [6, 28, 54, 80, 106],
        [6, 32, 58, 84, 110],
        [6, 30, 58, 86, 114],
        [6, 34, 62, 90, 118],
        [6, 26, 50, 74, 98, 122],
        [6, 30, 54, 78, 102, 126],
        [6, 26, 52, 78, 104, 130],
        [6, 30, 56, 82, 108, 134],
        [6, 34, 60, 86, 112, 138],
        [6, 30, 58, 86, 114, 142],
        [6, 34, 62, 90, 118, 146],
        [6, 30, 54, 78, 102, 126, 150],
        [6, 24, 50, 76, 102, 128, 154],
        [6, 28, 54, 80, 106, 132, 158],
        [6, 32, 58, 84, 110, 136, 162],
        [6, 26, 54, 82, 110, 138, 166],
        [6, 30, 58, 86, 114, 142, 170]
    ];

    // Error correction generator polynomials
    const G15 = 0b10100110111;
    const G18 = 0b1111100100101;
    const G15_MASK = 0b101010000010010;

    // Pad bytes
    const PAD0 = 0xEC;
    const PAD1 = 0x11;

    // Error correction levels mapping
    private static $errorCorrectionLevels = [
        'L' => 1,
        'M' => 0,
        'Q' => 3,
        'H' => 2
    ];

    public function __construct($typeNumber, $errorCorrectLevel) {
        $this->typeNumber = $typeNumber;
        $this->errorCorrectLevel = $errorCorrectLevel;
        $this->modules = null;
        $this->moduleCount = 0;
        $this->dataCache = null;
        $this->dataList = [];
    }

    public function addData($data, $mode = null) {
        if (!$mode) {
            if (preg_match('/^[0-9]*$/', $data)) {
                $mode = 1; // Numeric
            } elseif (preg_match('/^[0-9A-Z $%*+\-.\/:]*$/', $data)) {
                $mode = 2; // Alphanumeric
            } else {
                $mode = 4; // Byte
            }
        }

        switch ($mode) {
            case 1: // Numeric
                $this->dataList[] = $this->createNumericData($data);
                break;
            case 2: // Alphanumeric
                $this->dataList[] = $this->createAlphaNumericData($data);
                break;
            case 4: // Byte
                $this->dataList[] = $this->createByteData($data);
                break;
            case 8: // Kanji
                $this->dataList[] = $this->createKanjiData($data);
                break;
            default:
                throw new InvalidArgumentException("Invalid mode: $mode");
        }

        $this->dataCache = null;
    }

    private function createNumericData($data) {
        return new class($data) {
            private $data;

            public function __construct($data) {
                $this->data = $data;
            }

            public function getMode() {
                return 1; // Numeric mode
            }

            public function getLength() {
                return strlen($this->data);
            }

            public function write(&$buffer) {
                $data = $this->data;
                $i = 0;

                while ($i + 2 < strlen($data)) {
                    $num = intval(substr($data, $i, 3));
                    $buffer->put($num, 10);
                    $i += 3;
                }

                if ($i < strlen($data)) {
                    if (strlen($data) - $i == 1) {
                        $num = intval(substr($data, $i, 1));
                        $buffer->put($num, 4);
                    } else {
                        $num = intval(substr($data, $i, 2));
                        $buffer->put($num, 7);
                    }
                }
            }
        };
    }

    private function createAlphaNumericData($data) {
        return new class($data) {
            private $data;

            public function __construct($data) {
                $this->data = $data;
            }

            public function getMode() {
                return 2; // Alphanumeric mode
            }

            public function getLength() {
                return strlen($this->data);
            }

            public function write(&$buffer) {
                $data = $this->data;
                $i = 0;

                while ($i + 1 < strlen($data)) {
                    $char1 = $this->getCode($data[$i]);
                    $char2 = $this->getCode($data[$i + 1]);
                    $buffer->put($char1 * 45 + $char2, 11);
                    $i += 2;
                }

                if ($i < strlen($data)) {
                    $char = $this->getCode($data[$i]);
                    $buffer->put($char, 6);
                }
            }

            private function getCode($c) {
                if ('0' <= $c && $c <= '9') {
                    return ord($c) - ord('0');
                }
                if ('A' <= $c && $c <= 'Z') {
                    return ord($c) - ord('A') + 10;
                }

                switch ($c) {
                    case ' ': return 36;
                    case '$': return 37;
                    case '%': return 38;
                    case '*': return 39;
                    case '+': return 40;
                    case '-': return 41;
                    case '.': return 42;
                    case '/': return 43;
                    case ':': return 44;
                    default:
                        throw new Exception("illegal char: " . $c);
                }
            }
        };
    }

    private function createByteData($data) {
        return new class($data) {
            private $data;

            public function __construct($data) {
                $this->data = $data;
            }

            public function getMode() {
                return 4; // Byte mode
            }

            public function getLength() {
                return strlen($this->data);
            }

            public function write(&$buffer) {
                for ($i = 0; $i < strlen($this->data); $i++) {
                    $buffer->put(ord($this->data[$i]), 8);
                }
            }
        };
    }

    private function createKanjiData($data) {
        // Simplified Kanji implementation
        return new class($data) {
            private $data;

            public function __construct($data) {
                $this->data = $data;
            }

            public function getMode() {
                return 8; // Kanji mode
            }

            public function getLength() {
                return floor(strlen($this->data) / 2);
            }

            public function write(&$buffer) {
                $data = $this->data;
                $i = 0;

                while ($i + 1 < strlen($data)) {
                    $code = (ord($data[$i]) << 8) | ord($data[$i + 1]);

                    if (0x8140 <= $code && $code <= 0x9FFC) {
                        $code -= 0x8140;
                    } elseif (0xE040 <= $code && $code <= 0xEBBF) {
                        $code -= 0xC140;
                    } else {
                        throw new Exception("illegal char at " . ($i + 1) . "/" . $code);
                    }

                    $code = (($code >> 8) * 0xC0) + ($code & 0xFF);
                    $buffer->put($code, 13);
                    $i += 2;
                }

                if ($i < strlen($data)) {
                    throw new Exception("illegal char at " . ($i + 1));
                }
            }
        };
    }

    public function make() {
        if (empty($this->dataList)) {
            throw new RuntimeException("No data added to QR code");
        }

        // FIX: Auto-detect type number if set to 0
        if ($this->typeNumber === 0) {
            $this->typeNumber = $this->getBestTypeNumber();
        }

        $this->makeImpl(false, $this->getBestMaskPattern());
    }

    // Add this helper method to calculate the smallest version that fits the data
    private function getBestTypeNumber(): int
    {
        for ($type = 1; $type <= 40; $type++) {
            try {
                $rsBlocks = $this->getRSBlocks($type, $this->errorCorrectLevel);
                $buffer = new QRBitBuffer();

                // Try writing data to a buffer to check length
                foreach ($this->dataList as $data) {
                    $buffer->put($data->getMode(), 4);
                    // Length bits depend on the version (type)
                    $buffer->put($data->getLength(), $this->getLengthInBits($data->getMode(), $type));
                    $data->write($buffer);
                }

                // Calculate total capacity of this version
                $totalDataCount = 0;
                foreach ($rsBlocks as $block) {
                    $totalDataCount += $block['dataCount'];
                }

                // If data fits, return this version
                if ($buffer->getLengthInBits() <= $totalDataCount * 8) {
                    return $type;
                }
            } catch (Exception $e) {
                // Determine if error is due to overflow or invalid inputs,
                // but generally just continue to next version
                continue;
            }
        }

        throw new RuntimeException("Data too long for any QR version with current Error Correction Level");
    }

    private function makeImpl($test, $maskPattern) {
        $this->moduleCount = $this->typeNumber * 4 + 17;
        $this->modules = array_fill(0, $this->moduleCount, array_fill(0, $this->moduleCount, null));

        $this->setupPositionProbePattern(0, 0);
        $this->setupPositionProbePattern($this->moduleCount - 7, 0);
        $this->setupPositionProbePattern(0, $this->moduleCount - 7);
        $this->setupPositionAdjustPattern();
        $this->setupTimingPattern();
        $this->setupTypeInfo($test, $maskPattern);

        if ($this->typeNumber >= 7) {
            $this->setupTypeNumber($test);
        }

        if ($this->dataCache == null) {
            $this->dataCache = $this->createData($this->typeNumber, $this->errorCorrectLevel, $this->dataList);
        }

        $this->mapData($this->dataCache, $maskPattern);
    }

    private function setupPositionProbePattern($row, $col) {
        for ($r = -1; $r <= 7; $r++) {
            for ($c = -1; $c <= 7; $c++) {
                if ($row + $r <= -1 || $this->moduleCount <= $row + $r ||
                    $col + $c <= -1 || $this->moduleCount <= $col + $c) {
                    continue;
                }

                if ((0 <= $r && $r <= 6 && ($c == 0 || $c == 6)) ||
                    (0 <= $c && $c <= 6 && ($r == 0 || $r == 6)) ||
                    (2 <= $r && $r <= 4 && 2 <= $c && $c <= 4)) {
                    $this->modules[$row + $r][$col + $c] = true;
                } else {
                    $this->modules[$row + $r][$col + $c] = false;
                }
            }
        }
    }

    private function setupPositionAdjustPattern() {
        if ($this->typeNumber < 2) return;

        $pos = QRCodeGenerator::PATTERN_POSITION_TABLE[$this->typeNumber - 1];

        for ($i = 0; $i < count($pos); $i++) {
            for ($j = 0; $j < count($pos); $j++) {
                $row = $pos[$i];
                $col = $pos[$j];

                if ($this->modules[$row][$col] !== null) {
                    continue;
                }

                for ($r = -2; $r <= 2; $r++) {
                    for ($c = -2; $c <= 2; $c++) {
                        $this->modules[$row + $r][$col + $c] =
                            ($r == -2 || $r == 2 || $c == -2 || $c == 2 || ($r == 0 && $c == 0));
                    }
                }
            }
        }
    }

    private function setupTimingPattern() {
        for ($r = 8; $r < $this->moduleCount - 8; $r++) {
            if ($this->modules[$r][6] === null) {
                $this->modules[$r][6] = ($r % 2 == 0);
            }
        }

        for ($c = 8; $c < $this->moduleCount - 8; $c++) {
            if ($this->modules[6][$c] === null) {
                $this->modules[6][$c] = ($c % 2 == 0);
            }
        }
    }

    private function setupTypeInfo($test, $maskPattern) {
        $ecLevel = isset(self::$errorCorrectionLevels[$this->errorCorrectLevel])
            ? self::$errorCorrectionLevels[$this->errorCorrectLevel]
            : 2;

        $data = ($ecLevel << 3) | $maskPattern;
        $bits = $this->getBCHTypeInfo($data);

        for ($i = 0; $i < 15; $i++) {
            $mod = (!$test && (($bits >> $i) & 1) == 1);

            if ($i < 6) {
                $this->modules[$i][8] = $mod;
            } elseif ($i < 8) {
                $this->modules[$i + 1][8] = $mod;
            } else {
                $this->modules[$this->moduleCount - 15 + $i][8] = $mod;
            }
        }

        for ($i = 0; $i < 15; $i++) {
            $mod = (!$test && (($bits >> $i) & 1) == 1);

            if ($i < 8) {
                $this->modules[8][$this->moduleCount - $i - 1] = $mod;
            } elseif ($i < 9) {
                $this->modules[8][15 - $i - 1 + 1] = $mod;
            } else {
                $this->modules[8][15 - $i - 1] = $mod;
            }
        }

        $this->modules[$this->moduleCount - 8][8] = !$test;
    }

    private function setupTypeNumber($test) {
        $bits = $this->getBCHTypeNumber($this->typeNumber);

        for ($i = 0; $i < 18; $i++) {
            $mod = (!$test && (($bits >> $i) & 1) == 1);
            $this->modules[floor($i / 3)][$i % 3 + $this->moduleCount - 8 - 3] = $mod;
        }

        for ($i = 0; $i < 18; $i++) {
            $mod = (!$test && (($bits >> $i) & 1) == 1);
            $this->modules[$i % 3 + $this->moduleCount - 8 - 3][floor($i / 3)] = $mod;
        }
    }

    private function getBCHTypeInfo($data) {
        $d = $data << 10;
        while ($this->getBCHDigit($d) - $this->getBCHDigit(QRCodeGenerator::G15) >= 0) {
            $d ^= (QRCodeGenerator::G15 << ($this->getBCHDigit($d) - $this->getBCHDigit(QRCodeGenerator::G15)));
        }
        return (($data << 10) | $d) ^ QRCodeGenerator::G15_MASK;
    }

    private function getBCHTypeNumber($data) {
        $d = $data << 12;
        while ($this->getBCHDigit($d) - $this->getBCHDigit(QRCodeGenerator::G18) >= 0) {
            $d ^= (QRCodeGenerator::G18 << ($this->getBCHDigit($d) - $this->getBCHDigit(QRCodeGenerator::G18)));
        }
        return ($data << 12) | $d;
    }

    private function getBCHDigit($data) {
        $digit = 0;
        while ($data != 0) {
            $digit++;
            $data >>= 1;
        }
        return $digit;
    }

    private function mapData($data, $maskPattern) {
        $inc = -1;
        $row = $this->moduleCount - 1;
        $bitIndex = 7;
        $byteIndex = 0;

        for ($col = $this->moduleCount - 1; $col > 0; $col -= 2) {
            if ($col == 6) $col--;

            while (true) {
                for ($c = 0; $c < 2; $c++) {
                    if ($this->modules[$row][$col - $c] === null) {
                        $dark = false;

                        if ($byteIndex < count($data)) {
                            $dark = ((($data[$byteIndex] >> $bitIndex) & 1) == 1);
                        }

                        $mask = $this->getMask($maskPattern, $row, $col - $c);

                        if ($mask) {
                            $dark = !$dark;
                        }

                        $this->modules[$row][$col - $c] = $dark;
                        $bitIndex--;

                        if ($bitIndex == -1) {
                            $byteIndex++;
                            $bitIndex = 7;
                        }
                    }
                }

                $row += $inc;

                if ($row < 0 || $this->moduleCount <= $row) {
                    $row -= $inc;
                    $inc = -$inc;
                    break;
                }
            }
        }
    }

    private function getMask($maskPattern, $i, $j) {
        switch ($maskPattern) {
            case 0: return ($i + $j) % 2 == 0;
            case 1: return $i % 2 == 0;
            case 2: return $j % 3 == 0;
            case 3: return ($i + $j) % 3 == 0;
            case 4: return (floor($i / 2) + floor($j / 3)) % 2 == 0;
            case 5: return ($i * $j) % 2 + ($i * $j) % 3 == 0;
            case 6: return (($i * $j) % 2 + ($i * $j) % 3) % 2 == 0;
            case 7: return (($i * $j) % 3 + ($i + $j) % 2) % 2 == 0;
            default:
                throw new Exception("bad maskPattern: " . $maskPattern);
        }
    }

    private function getBestMaskPattern() {
        $minLostPoint = 0;
        $pattern = 0;

        for ($i = 0; $i < 8; $i++) {
            $this->makeImpl(true, $i);
            $lostPoint = $this->getLostPoint();

            if ($i == 0 || $minLostPoint > $lostPoint) {
                $minLostPoint = $lostPoint;
                $pattern = $i;
            }
        }

        return $pattern;
    }

    private function getLostPoint() {
        $lostPoint = 0;

        // Level 1
        for ($row = 0; $row < $this->moduleCount; $row++) {
            for ($col = 0; $col < $this->moduleCount; $col++) {
                $sameCount = 0;
                $dark = $this->modules[$row][$col];

                for ($r = -1; $r <= 1; $r++) {
                    for ($c = -1; $c <= 1; $c++) {
                        if ($r == 0 && $c == 0) {
                            continue;
                        }
                        if ($row + $r < 0 || $this->moduleCount <= $row + $r ||
                            $col + $c < 0 || $this->moduleCount <= $col + $c) {
                            continue;
                        }

                        if ($dark == $this->modules[$row + $r][$col + $c]) {
                            $sameCount++;
                        }
                    }
                }

                if ($sameCount > 5) {
                    $lostPoint += (3 + $sameCount - 5);
                }
            }
        }

        // Level 2
        for ($row = 0; $row < $this->moduleCount - 1; $row++) {
            for ($col = 0; $col < $this->moduleCount - 1; $col++) {
                $count = 0;
                if ($this->modules[$row][$col]) $count++;
                if ($this->modules[$row + 1][$col]) $count++;
                if ($this->modules[$row][$col + 1]) $count++;
                if ($this->modules[$row + 1][$col + 1]) $count++;
                if ($count == 0 || $count == 4) {
                    $lostPoint += 3;
                }
            }
        }

        // Level 3
        for ($row = 0; $row < $this->moduleCount; $row++) {
            for ($col = 0; $col < $this->moduleCount - 6; $col++) {
                if ($this->modules[$row][$col] &&
                    !$this->modules[$row][$col + 1] &&
                    $this->modules[$row][$col + 2] &&
                    $this->modules[$row][$col + 3] &&
                    $this->modules[$row][$col + 4] &&
                    !$this->modules[$row][$col + 5] &&
                    $this->modules[$row][$col + 6]) {
                    $lostPoint += 40;
                }
            }
        }

        for ($col = 0; $col < $this->moduleCount; $col++) {
            for ($row = 0; $row < $this->moduleCount - 6; $row++) {
                if ($this->modules[$row][$col] &&
                    !$this->modules[$row + 1][$col] &&
                    $this->modules[$row + 2][$col] &&
                    $this->modules[$row + 3][$col] &&
                    $this->modules[$row + 4][$col] &&
                    !$this->modules[$row + 5][$col] &&
                    $this->modules[$row + 6][$col]) {
                    $lostPoint += 40;
                }
            }
        }

        // Level 4
        $darkCount = 0;
        for ($col = 0; $col < $this->moduleCount; $col++) {
            for ($row = 0; $row < $this->moduleCount; $row++) {
                if ($this->modules[$row][$col]) {
                    $darkCount++;
                }
            }
        }

        $ratio = abs(100 * $darkCount / $this->moduleCount / $this->moduleCount - 50) / 5;
        $lostPoint += $ratio * 10;

        return $lostPoint;
    }

    private function createData($typeNumber, $errorCorrectLevel, $dataList) {
        $rsBlocks = $this->getRSBlocks($typeNumber, $errorCorrectLevel);
        $buffer = new QRBitBuffer();

        for ($i = 0; $i < count($dataList); $i++) {
            $data = $dataList[$i];
            $buffer->put($data->getMode(), 4);
            $buffer->put($data->getLength(), $this->getLengthInBits($data->getMode(), $typeNumber));
            $data->write($buffer);
        }

        $totalDataCount = 0;
        for ($i = 0; $i < count($rsBlocks); $i++) {
            $totalDataCount += $rsBlocks[$i]['dataCount'];
        }

        if ($buffer->getLengthInBits() > $totalDataCount * 8) {
            throw new Exception("code length overflow. (" . $buffer->getLengthInBits() . ">" . ($totalDataCount * 8) . ")");
        }

        if ($buffer->getLengthInBits() + 4 <= $totalDataCount * 8) {
            $buffer->put(0, 4);
        }

        while ($buffer->getLengthInBits() % 8 != 0) {
            $buffer->putBit(false);
        }

        while (true) {
            if ($buffer->getLengthInBits() >= $totalDataCount * 8) {
                break;
            }
            $buffer->put(QRCodeGenerator::PAD0, 8);

            if ($buffer->getLengthInBits() >= $totalDataCount * 8) {
                break;
            }
            $buffer->put(QRCodeGenerator::PAD1, 8);
        }

        return $this->createBytes($buffer, $rsBlocks);
    }

    private function createBytes($buffer, $rsBlocks) {
        $offset = 0;
        $maxDcCount = 0;
        $maxEcCount = 0;
        $dcdata = array_fill(0, count($rsBlocks), []);
        $ecdata = array_fill(0, count($rsBlocks), []);

        for ($r = 0; $r < count($rsBlocks); $r++) {
            $dcCount = $rsBlocks[$r]['dataCount'];
            $ecCount = $rsBlocks[$r]['totalCount'] - $dcCount;

            $maxDcCount = max($maxDcCount, $dcCount);
            $maxEcCount = max($maxEcCount, $ecCount);

            $dcdata[$r] = array_fill(0, $dcCount, 0);
            for ($i = 0; $i < count($dcdata[$r]); $i++) {
                $dcdata[$r][$i] = 0xFF & $buffer->getBuffer()[$i + $offset];
            }
            $offset += $dcCount;

            $rsPoly = $this->getErrorCorrectPolynomial($ecCount);
            $rawPoly = new QRPolynomial($dcdata[$r], $rsPoly->getLength() - 1);
            $modPoly = $rawPoly->mod($rsPoly);

            $ecdata[$r] = array_fill(0, $rsPoly->getLength() - 1, 0);
            for ($i = 0; $i < count($ecdata[$r]); $i++) {
                $modIndex = $i + $modPoly->getLength() - count($ecdata[$r]);
                $ecdata[$r][$i] = ($modIndex >= 0) ? $modPoly->getAt($modIndex) : 0;
            }
        }

        $totalCodeCount = 0;
        for ($i = 0; $i < count($rsBlocks); $i++) {
            $totalCodeCount += $rsBlocks[$i]['totalCount'];
        }

        $data = array_fill(0, $totalCodeCount, 0);
        $index = 0;

        for ($i = 0; $i < $maxDcCount; $i++) {
            for ($r = 0; $r < count($rsBlocks); $r++) {
                if ($i < count($dcdata[$r])) {
                    $data[$index++] = $dcdata[$r][$i];
                }
            }
        }

        for ($i = 0; $i < $maxEcCount; $i++) {
            for ($r = 0; $r < count($rsBlocks); $r++) {
                if ($i < count($ecdata[$r])) {
                    $data[$index++] = $ecdata[$r][$i];
                }
            }
        }

        return $data;
    }

    private function getLengthInBits($mode, $type) {
        if (1 <= $type && $type < 10) {
            switch ($mode) {
                case 1: return 10;
                case 2: return 9;
                case 4: return 8;
                case 8: return 8;
                default: throw new Exception("mode: " . $mode);
            }
        } elseif ($type < 27) {
            switch ($mode) {
                case 1: return 12;
                case 2: return 11;
                case 4: return 16;
                case 8: return 10;
                default: throw new Exception("mode: " . $mode);
            }
        } elseif ($type < 41) {
            switch ($mode) {
                case 1: return 14;
                case 2: return 13;
                case 4: return 16;
                case 8: return 12;
                default: throw new Exception("mode: " . $mode);
            }
        } else {
            throw new Exception("type: " . $type);
        }
    }

    private function getErrorCorrectPolynomial($errorCorrectLength) {
        $a = new QRPolynomial([1], 0);

        for ($i = 0; $i < $errorCorrectLength; $i++) {
            $a = $a->multiply(new QRPolynomial([1, QRPolynomial::gexp($i)], 0));
        }

        return $a;
    }

    private function getRSBlocks($typeNumber, $errorCorrectLevel) {
        $rsBlock = $this->getRsBlockTable($typeNumber, $errorCorrectLevel);

        if ($rsBlock === null) {
            throw new Exception("bad rs block @ typeNumber:" . $typeNumber . "/errorCorrectionLevel:" . $errorCorrectLevel);
        }

        $length = count($rsBlock) / 3;
        $list = [];

        for ($i = 0; $i < $length; $i++) {
            $count = $rsBlock[$i * 3 + 0];
            $totalCount = $rsBlock[$i * 3 + 1];
            $dataCount = $rsBlock[$i * 3 + 2];

            for ($j = 0; $j < $count; $j++) {
                $list[] = [
                    'totalCount' => $totalCount,
                    'dataCount' => $dataCount
                ];
            }
        }

        return $list;
    }

    private function getRsBlockTable($typeNumber, $errorCorrectLevel) {
        static $RS_BLOCK_TABLE = [
            [1, 26, 19],
            [1, 26, 16],
            [1, 26, 13],
            [1, 26, 9],
            [1, 44, 34],
            [1, 44, 28],
            [1, 44, 22],
            [1, 44, 16],
            [1, 70, 55],
            [1, 70, 44],
            [2, 35, 17],
            [2, 35, 13],
            [1, 100, 80],
            [2, 50, 32],
            [2, 50, 24],
            [4, 25, 9],
            [1, 134, 108],
            [2, 67, 43],
            [2, 33, 15, 2, 34, 16],
            [2, 33, 11, 2, 34, 12],
            [2, 86, 68],
            [4, 43, 27],
            [4, 43, 19],
            [4, 43, 15],
            [2, 98, 78],
            [4, 49, 31],
            [2, 32, 14, 4, 33, 15],
            [4, 39, 13, 1, 40, 14],
            [2, 121, 97],
            [2, 60, 38, 2, 61, 39],
            [4, 40, 18, 2, 41, 19],
            [4, 40, 14, 2, 41, 15],
            [2, 146, 116],
            [3, 58, 36, 2, 59, 37],
            [4, 36, 16, 4, 37, 17],
            [4, 36, 12, 4, 37, 13],
            [2, 86, 68, 2, 87, 69],
            [4, 69, 43, 1, 70, 44],
            [6, 43, 19, 2, 44, 20],
            [6, 43, 15, 2, 44, 16],
            [4, 101, 81],
            [1, 80, 50, 4, 81, 51],
            [4, 50, 22, 4, 51, 23],
            [3, 36, 12, 8, 37, 13],
            [2, 116, 92, 2, 117, 93],
            [6, 58, 36, 2, 59, 37],
            [4, 46, 20, 6, 47, 21],
            [7, 42, 14, 4, 43, 15],
            [4, 133, 107],
            [8, 59, 37, 1, 60, 38],
            [8, 44, 20, 4, 45, 21],
            [12, 33, 11, 4, 34, 12],
            [3, 145, 115, 1, 146, 116],
            [4, 64, 40, 5, 65, 41],
            [11, 36, 16, 5, 37, 17],
            [11, 36, 12, 5, 37, 13],
            [5, 109, 87, 1, 110, 88],
            [5, 65, 41, 5, 66, 42],
            [5, 54, 24, 7, 55, 25],
            [11, 36, 12, 7, 37, 13],
            [5, 122, 98, 1, 123, 99],
            [7, 73, 45, 3, 74, 46],
            [15, 43, 19, 2, 44, 20],
            [3, 45, 15, 13, 46, 16],
            [1, 135, 107, 5, 136, 108],
            [10, 74, 46, 1, 75, 47],
            [1, 50, 22, 15, 51, 23],
            [2, 42, 14, 17, 43, 15],
            [5, 150, 120, 1, 151, 121],
            [9, 69, 43, 4, 70, 44],
            [17, 50, 22, 1, 51, 23],
            [2, 42, 14, 19, 43, 15],
            [3, 141, 113, 4, 142, 114],
            [3, 70, 44, 11, 71, 45],
            [17, 47, 21, 4, 48, 22],
            [9, 39, 13, 16, 40, 14],
            [3, 135, 107, 5, 136, 108],
            [3, 67, 41, 13, 68, 42],
            [15, 54, 24, 5, 55, 25],
            [15, 43, 15, 10, 44, 16],
            [4, 144, 116, 4, 145, 117],
            [17, 68, 42],
            [17, 50, 22, 6, 51, 23],
            [19, 46, 16, 6, 47, 17],
            [2, 139, 111, 7, 140, 112],
            [17, 74, 46],
            [7, 54, 24, 16, 55, 25],
            [34, 37, 13],
            [4, 151, 121, 5, 152, 122],
            [4, 75, 47, 14, 76, 48],
            [11, 54, 24, 14, 55, 25],
            [16, 45, 15, 14, 46, 16],
            [6, 147, 117, 4, 148, 118],
            [6, 73, 45, 14, 74, 46],
            [11, 54, 24, 16, 55, 25],
            [30, 46, 16, 2, 47, 17],
            [8, 132, 106, 4, 133, 107],
            [8, 75, 47, 13, 76, 48],
            [7, 54, 24, 22, 55, 25],
            [22, 45, 15, 13, 46, 16],
            [10, 142, 114, 2, 143, 115],
            [19, 74, 46, 4, 75, 47],
            [28, 50, 22, 6, 51, 23],
            [33, 46, 16, 4, 47, 17],
            [8, 152, 122, 4, 153, 123],
            [22, 73, 45, 3, 74, 46],
            [8, 53, 23, 26, 54, 24],
            [12, 45, 15, 28, 46, 16],
            [3, 147, 117, 10, 148, 118],
            [3, 73, 45, 23, 74, 46],
            [4, 54, 24, 31, 55, 25],
            [11, 45, 15, 31, 46, 16],
            [7, 146, 116, 7, 147, 117],
            [21, 73, 45, 7, 74, 46],
            [1, 53, 23, 37, 54, 24],
            [19, 45, 15, 26, 46, 16],
            [5, 145, 115, 10, 146, 116],
            [19, 75, 47, 10, 76, 48],
            [15, 54, 24, 25, 55, 25],
            [23, 45, 15, 25, 46, 16],
            [13, 145, 115, 3, 146, 116],
            [2, 74, 46, 29, 75, 47],
            [42, 54, 24, 1, 55, 25],
            [23, 45, 15, 28, 46, 16],
            [17, 145, 115],
            [10, 74, 46, 23, 75, 47],
            [10, 54, 24, 35, 55, 25],
            [19, 45, 15, 35, 46, 16],
            [17, 145, 115, 1, 146, 116],
            [14, 74, 46, 21, 75, 47],
            [29, 54, 24, 19, 55, 25],
            [11, 45, 15, 46, 46, 16],
            [13, 145, 115, 6, 146, 116],
            [14, 74, 46, 23, 75, 47],
            [44, 54, 24, 7, 55, 25],
            [59, 46, 16, 1, 47, 17],
            [12, 151, 121, 7, 152, 122],
            [12, 75, 47, 26, 76, 48],
            [39, 54, 24, 14, 55, 25],
            [22, 45, 15, 41, 46, 16],
            [6, 151, 121, 14, 152, 122],
            [6, 75, 47, 34, 76, 48],
            [46, 54, 24, 10, 55, 25],
            [2, 45, 15, 64, 46, 16],
            [17, 152, 122, 4, 153, 123],
            [29, 74, 46, 14, 75, 47],
            [49, 54, 24, 10, 55, 25],
            [24, 45, 15, 46, 46, 16],
            [4, 152, 122, 18, 153, 123],
            [13, 74, 46, 32, 75, 47],
            [48, 54, 24, 14, 55, 25],
            [42, 45, 15, 32, 46, 16],
            [20, 147, 117, 4, 148, 118],
            [40, 75, 47, 7, 76, 48],
            [43, 54, 24, 22, 55, 25],
            [10, 45, 15, 67, 46, 16],
            [19, 148, 118, 6, 149, 119],
            [18, 75, 47, 31, 76, 48],
            [34, 54, 24, 34, 55, 25],
            [20, 45, 15, 61, 46, 16]
        ];

        switch ($errorCorrectLevel) {
            case 'L': return $RS_BLOCK_TABLE[($typeNumber - 1) * 4 + 0];
            case 'M': return $RS_BLOCK_TABLE[($typeNumber - 1) * 4 + 1];
            case 'Q': return $RS_BLOCK_TABLE[($typeNumber - 1) * 4 + 2];
            case 'H': return $RS_BLOCK_TABLE[($typeNumber - 1) * 4 + 3];
            default: return null;
        }
    }

    public function isDark($row, $col) {
        if ($row < 0 || $this->moduleCount <= $row || $col < 0 || $this->moduleCount <= $col) {
            throw new Exception($row . "," . $col);
        }
        return $this->modules[$row][$col];
    }

    public function getModuleCount() {
        return $this->moduleCount;
    }
}

class QRBitBuffer {
    private $buffer = [];
    private $length = 0;

    public function getBuffer() {
        return $this->buffer;
    }

    public function getAt($index) {
        $bufIndex = floor($index / 8);
        if (!isset($this->buffer[$bufIndex])) {
            return false;
        }
        return (($this->buffer[$bufIndex] >> (7 - $index % 8)) & 1) == 1;
    }

    public function put($num, $length) {
        for ($i = 0; $i < $length; $i++) {
            $this->putBit((($num >> ($length - $i - 1)) & 1) == 1);
        }
    }

    public function getLengthInBits() {
        return $this->length;
    }

    public function putBit($bit) {
        $bufIndex = floor($this->length / 8);
        if (count($this->buffer) <= $bufIndex) {
            $this->buffer[] = 0;
        }

        if ($bit) {
            $this->buffer[$bufIndex] |= (0x80 >> ($this->length % 8));
        }

        $this->length++;
    }
}

class QRPolynomial {
    private $num;

    // Galois field tables
    private static $expTable = null;
    private static $logTable = null;

    private static function initTables() {
        if (self::$expTable === null) {
            self::$expTable = array_fill(0, 256, 0);
            self::$logTable = array_fill(0, 256, 0);

            for ($i = 0; $i < 8; $i++) {
                self::$expTable[$i] = 1 << $i;
            }

            for ($i = 8; $i < 256; $i++) {
                self::$expTable[$i] = self::$expTable[$i - 4] ^
                    self::$expTable[$i - 5] ^
                    self::$expTable[$i - 6] ^
                    self::$expTable[$i - 8];
            }

            for ($i = 0; $i < 255; $i++) {
                self::$logTable[self::$expTable[$i]] = $i;
            }
        }
    }

    public function __construct($num, $shift = 0) {
        self::initTables();

        $offset = 0;
        while ($offset < count($num) && $num[$offset] == 0) {
            $offset++;
        }

        $this->num = array_fill(0, count($num) - $offset + $shift, 0);
        for ($i = 0; $i < count($num) - $offset; $i++) {
            $this->num[$i] = $num[$i + $offset];
        }
    }

    public function getAt($index) {
        return $this->num[$index];
    }

    public function getLength() {
        return count($this->num);
    }

    public function multiply($e) {
        $num = array_fill(0, $this->getLength() + $e->getLength() - 1, 0);

        for ($i = 0; $i < $this->getLength(); $i++) {
            for ($j = 0; $j < $e->getLength(); $j++) {
                $num[$i + $j] ^= self::gexp(self::glog($this->getAt($i)) + self::glog($e->getAt($j)));
            }
        }

        return new QRPolynomial($num, 0);
    }

    public function mod($e) {
        if ($this->getLength() - $e->getLength() < 0) {
            return $this;
        }

        $ratio = self::glog($this->getAt(0)) - self::glog($e->getAt(0));
        $num = array_fill(0, $this->getLength(), 0);

        for ($i = 0; $i < $this->getLength(); $i++) {
            $num[$i] = $this->getAt($i);
        }

        for ($i = 0; $i < $e->getLength(); $i++) {
            $num[$i] ^= self::gexp(self::glog($e->getAt($i)) + $ratio);
        }

        $newPoly = new QRPolynomial($num, 0);
        return $newPoly->mod($e);
    }

    public static function glog($n) {
        self::initTables();

        if ($n < 1) {
            throw new Exception("glog(" . $n . ")");
        }

        return self::$logTable[$n];
    }

    public static function gexp($n) {
        self::initTables();

        while ($n < 0) {
            $n += 255;
        }

        while ($n >= 256) {
            $n -= 255;
        }

        return self::$expTable[$n];
    }
}