<?php
// universal-smtp-api.php
// Universal SMTP Mail API using PHPMailer (Composer install only — no legacy autoloader).
// POST JSON with { smtp: {...}, message: {...}, debug?: {level?:0..4} }
// Always returns HTTP 200; errors are in the JSON so proxies don't mask them as 500/502.
// -----------------------------------------------------------------------------
// INSTALL:
//   composer require phpmailer/phpmailer vlucas/phpdotenv
// USAGE (curl at bottom). Optional Bearer auth via MAIL_API_KEY env.
// -----------------------------------------------------------------------------

declare(strict_types=1);

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception as MailException;

require __DIR__ . '/vendor/autoload.php';

// ---------- Headers / CORS --------------------------------------------------
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { echo json_encode(['ok'=>false,'error'=>'Use POST']); exit; }

// ---------- Optional API key (Bearer) --------------------------------------
$expectedKey = getenv('MAIL_API_KEY') ?: '1';
if ($expectedKey !== '') {
    $auth = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $pref = 'Bearer ';
    if (stripos($auth, $pref) !== 0 || substr($auth, strlen($pref)) !== $expectedKey) {
        echo json_encode(['ok'=>false,'error'=>'Unauthorized']);
        exit;
    }
}

// ---------- Read JSON -------------------------------------------------------
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);
if (!is_array($in)) { echo json_encode(['ok'=>false,'error'=>'Invalid JSON body']); exit; }

$debug = $in['debug'] ?? [];
$debugLevel = (int)($debug['level'] ?? 0); // 0..4 (4 = very verbose)

$smtp = $in['smtp'] ?? [];
$msg  = $in['message'] ?? [];

// ---------- Limits ----------------------------------------------------------
$MAX_RECIPIENTS = 50;
$MAX_BODY = 400000;                // ~400 KB each for html/text
$MAX_ATTACH = 10;                  // max number of attachments
$MAX_ATTACH_B = 8 * 1024 * 1024;   // 8 MB per attachment

// ---------- Helpers ---------------------------------------------------------
$validateEmail = function(string $s): ?string {
    $s = trim($s);
    if ($s === '') return null;
    if (preg_match('/<([^>]+)>$/', $s, $m)) { $s = $m[1]; }
    return filter_var($s, FILTER_VALIDATE_EMAIL) ?: null;
};

$normList = function($arr) use ($validateEmail): array {
    if (!is_array($arr)) return [];
    $out = [];
    foreach ($arr as $x) if (is_string($x)) { $e = $validateEmail($x); if ($e) $out[] = $e; }
    return array_values(array_unique($out));
};

// ---------- Extract SMTP ----------------------------------------------------
$host      = trim((string)($smtp['host'] ?? ''));
$port      = (int)($smtp['port'] ?? 587);
$secureStr = strtolower((string)($smtp['secure'] ?? 'tls')); // tls|ssl|starttls|none
$username  = (string)($smtp['username'] ?? '');
$password  = (string)($smtp['password'] ?? '');
$auth      = array_key_exists('auth', $smtp) ? (bool)$smtp['auth'] : true;
$timeout   = (int)($smtp['timeout'] ?? 30);
$verifyTls = array_key_exists('verify_tls', $smtp) ? (bool)$smtp['verify_tls'] : true; // keep true in prod

// ---------- Extract Message -------------------------------------------------
$from      = (string)($msg['from'] ?? ''); // "Name <email>" or email
$fromNameI = (string)($msg['from_name'] ?? '');
$reply_to  = (string)($msg['reply_to'] ?? '');
$to        = $msg['to'] ?? [];
$cc        = $msg['cc'] ?? [];
$bcc       = $msg['bcc'] ?? [];
$subject   = (string)($msg['subject'] ?? '');
$html      = (string)($msg['html'] ?? '');
$text      = (string)($msg['text'] ?? '');
$headers   = $msg['headers'] ?? [];           // [{name,value}]
$attachments = $msg['attachments'] ?? [];     // [{filename, content(base64)|path, type}]

// ---------- Validate --------------------------------------------------------
$errs = [];
$toList  = $normList($to);
$ccList  = $normList($cc);
$bccList = $normList($bcc);

if ($host === '') $errs[] = 'smtp.host is required';
if (empty($toList)) $errs[] = 'message.to must contain at least one valid email';
if ($subject === '') $errs[] = 'message.subject is required';
if ($html === '' && $text === '') $errs[] = 'Provide message.html or message.text';
if (mb_strlen($html) > $MAX_BODY || mb_strlen($text) > $MAX_BODY) $errs[] = 'Body too large';
if ((count($toList) + count($ccList) + count($bccList)) > $MAX_RECIPIENTS) $errs[] = 'Too many recipients';
if (!empty($attachments) && count($attachments) > $MAX_ATTACH) $errs[] = 'Too many attachments';

if ($errs) { echo json_encode(['ok'=>false,'error'=>'Validation failed','details'=>$errs]); exit; }

// Derive From
$fromEmail = '';
$fromName  = $fromNameI;
if ($from !== '') {
    if (preg_match('/^(.*?)<([^>]+)>$/', $from, $m)) {
        $fromName  = trim($m[1], " \"\t");
        $fromEmail = $validateEmail($m[2]) ?: '';
    } else {
        $fromEmail = $validateEmail($from) ?: '';
    }
}
if ($fromEmail === '') $fromEmail = $username ?: 'no-reply@localhost';
if ($fromName  === '') $fromName  = 'Mail API';

// Capture SMTP debug without leaking secrets
$debugBuffer = '';
$debugCollector = function($str) use (&$debugBuffer, $username, $password) {
    $s = (string)$str;
    if ($username !== '') $s = str_replace($username, '[REDACTED-USER]', $s);
    if ($password !== '') $s = str_replace($password, '[REDACTED-PASS]', $s);
    $debugBuffer .= $s;
};

try {
    $mail = new PHPMailer(true);
    $mail->isSMTP();
    $mail->Host = $host;
    $mail->Port = $port ?: 587;
    $mail->SMTPAuth = $auth;
    if ($auth) { $mail->Username = $username; $mail->Password = $password; }

    if ($secureStr === 'ssl') {
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
    } elseif ($secureStr === 'tls' || $secureStr === 'starttls') {
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
    } elseif ($secureStr === 'none' || $secureStr === '') {
        $mail->SMTPSecure = '';
    } else {
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
    }

    $mail->Timeout = max(5, $timeout);
    $mail->SMTPOptions = ['ssl' => [
        'verify_peer' => $verifyTls,
        'verify_peer_name' => $verifyTls,
        'allow_self_signed' => false,
    ]];

    // Debugging
    $mail->SMTPDebug = $debugLevel;        // 0..4
    $mail->Debugoutput = $debugCollector;  // capture to JSON-safe buffer

    $mail->setFrom($fromEmail, $fromName);
    foreach ($toList as $r) { $mail->addAddress($r); }
    foreach ($ccList as $r) { $mail->addCC($r); }
    foreach ($bccList as $r) { $mail->addBCC($r); }

    if ($reply_to) { $rt = $validateEmail($reply_to); if ($rt) $mail->addReplyTo($rt); }

    if (is_array($headers)) {
        foreach ($headers as $h) {
            if (is_array($h) && isset($h['name'], $h['value'])) {
                $mail->addCustomHeader((string)$h['name'], (string)$h['value']);
            }
        }
    }

    $mail->isHTML($html !== '');
    $mail->Subject = $subject;
    if ($html !== '') $mail->Body = $html;
    if ($text !== '') $mail->AltBody = $text;

    if (is_array($attachments)) {
        $i = 0;
        foreach ($attachments as $a) {
            if (++$i > $MAX_ATTACH) break;
            if (!is_array($a)) continue;
            $filename = (string)($a['filename'] ?? 'attachment');
            if (!empty($a['path'])) {
                $path = (string)$a['path'];
                if (is_readable($path)) $mail->addAttachment($path, $filename);
                continue;
            }
            $contentB64 = (string)($a['content'] ?? '');
            $type       = (string)($a['type'] ?? 'application/octet-stream');
            if ($contentB64 !== '') {
                $bin = base64_decode($contentB64, true);
                if ($bin !== false) {
                    if (strlen($bin) > $MAX_ATTACH_B) throw new MailException('Attachment too large');
                    $mail->addStringAttachment($bin, $filename, 'base64', $type);
                }
            }
        }
    }

    $ok = $mail->send();
    echo json_encode(['ok'=>$ok,'message'=>$ok?'mail sent':'mail not sent','debug'=>($debugLevel? $debugBuffer: null)]);
}
catch (MailException $e) {
    echo json_encode(['ok'=>false,'error'=>'Mailer error','details'=>$e->getMessage(),'debug'=>($debugLevel? $debugBuffer: null)]);
}
catch (\Throwable $e) {
    echo json_encode(['ok'=>false,'error'=>'Server error','details'=>$e->getMessage(),'debug'=>($debugLevel? $debugBuffer: null)]);
}

/*
Example curl (replace values):

curl -X POST https://your.host/universal-smtp-api.php \
  -H "Authorization: Bearer $MAIL_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "debug": {"level": 2},
    "smtp": {
      "host": "65.108.74.236",
      "port": 587,
      "secure": "tls",
      "auth": true,
      "username": "no-reply@meghait.in",
      "password": "REDACTED",
      "verify_tls": true,
      "timeout": 30
    },
    "message": {
      "from": "Megha Enterprises <no-reply@meghait.in>",
      "reply_to": "support@meghait.in",
      "to": ["princerk742@gmail.com"],
      "subject": "Complaint",
      "html": "<strong>Hello</strong> from the universal API",
      "text": "Hello from the universal API"
    }
  }'
*/
